import { MystiqueManifestFragment } from 'mystique/types/manifest';
import gql from 'graphql-tag';
import { parse } from '../src/utils/GqlUtils';

export const manifest: MystiqueManifestFragment = {
  manifestVersion: '2.0',
  settings: [
    {
      name: 'fc.order.list.status.column',
    },
  ],
    routes: [
        {
          type: 'section',
          nav: {
            label: 'i18n:fc.om.nav',
            icon: 'shopping_cart',
          },
            pages: [
                {
                  type: 'page',
                  path: 'orders',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query ($orders_first: Int) {
                              orders(first: $orders_first) {
                                edges {
                                  node {
                                    id
                                    ref
                                    retailer {
                                      id
                                      tradingName
                                    }
                                    type
                                    status
                                    retailer {
                                      id
                                    }
                                    workflowRef
                                    workflowVersion
                                    totalPrice
                                    totalTaxPrice
                                    items {
                                      edges {
                                        node {
                                          currency
                                        }
                                      }
                                    }
                                    createdOn
                                    customer {
                                      id
                                      firstName
                                      lastName
                                    }
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        orders_first: 100,
                      },
                    },
                  nav: {
                    label: 'i18n:fc.om.orders.index.nav',
                    icon: 'library_books',
                  },
                  props: {
                    title: 'i18n:fc.om.orders.index.title',
                    actions: {
                      primary: [
                        {
                          type: 'mutation',
                          label: 'i18n:fc.se.field.create.order.user.action.label',
                          name: 'createComment',
                          filter: {
                            type: 'include',
                            names: ['text'],
                          },
                          args: {
                            entityId: '-1',
                            entityType:
                              'i18n:fc.om.orders.index.userAction.entityType.order',
                          },
                          overrides: {
                            text: {
                              component: 'createOrderWizard',
                            },
                          },
                        },
                      ],
                    },
                  },
                  descendants: [
                    {
                      component: 'fc.list',
                      props: {
                        defaultPageSize: 100,
                        dataSource: 'orders',
                        responsiveness: 'card',
                        filter: {
                          enabled: true,
                          exclude: [
                            'workflowRef',
                            'workflowVersion',
                            'totalPrice',
                            'totalTaxPrice',
                          ],
                        },
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.orders.index.list.column.orderRef.heading',
                            template: '{{node.ref}}',
                            link: '#/orders/{{node.id}}/{{node.retailer.id}}/{{node.ref}}',
                            condition: '{{and node.id node.retailer.id node.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orders.index.list.column.customer.heading',
                            template:
                              '{{node.customer.firstName}} {{node.customer.lastName}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orders.index.list.column.orderType.heading',
                            template: '{{node.type}}',
                          },
                          {
                            label: 'i18n:fc.om.orders.index.list.column.status.heading',
                            type: 'component',
                            options: {
                              component: 'fc.attribute.column',
                              props: {
                                contentColumnSetting: 'fc.order.list.status.column',
                                value: '{{status}}',
                              },
                              dataSource: 'node',
                            },
                          },
                          {
                            label:
                              'i18n:fc.om.orders.index.list.column.orderValue.heading',
                            template:
                              '{{currency node.totalPrice node.items.edges.0.node.currency}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orders.index.list.column.createdOn.heading',
                            template:
                              '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'orders/:id/:retailerId/:ref',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $id: ID!
                              $items_after: String
                              $items_first: Int
                              $fulfilments_first: Int
                              $comments_first: Int
                              $financialTransactions_first: Int
                              $returnOrders_first: Int
                              $ref: String!
                              $retailerId: ID!
                            ) {
                              orderById(id: $id) {
                                id
                                __typename
                                ref
                                status
                                type
                                createdOn
                                updatedOn
                                retailer {
                                  id
                                  tradingName
                                }
                                customer {
                                  id
                                  ref
                                  firstName
                                  lastName
                                  primaryEmail
                                  primaryPhone
                                  timezone
                                }
                                items(after: $items_after, first: $items_first) {
                                  edges {
                                    node {
                                      id
                                      ref
                                      price
                                      quantity
                                      totalPrice
                                      currency
                                      product {
                                        name
                                      }
                                    }
                                  }
                                }
                                fulfilmentChoice {
                                  id
                                  pickupLocationRef
                                  deliveryAddress {
                                    id
                                    name
                                    street
                                    city
                                    state
                                    postcode
                                    country
                                    longitude
                                    latitude
                                    companyName
                                  }
                                  deliveryInstruction
                                  deliveryType
                                  fulfilmentPrice
                                }
                                fulfilments(first: $fulfilments_first) {
                                  edges {
                                    node {
                                      id
                                      deliveryType
                                      status
                                      eta
                                      createdOn
                                      fromAddress {
                                        ref
                                        companyName
                                        name
                                      }
                                      toAddress {
                                        ref
                                        companyName
                                        name
                                      }
                                    }
                                  }
                                }
                                financialTransactions(first: $financialTransactions_first) {
                                  edges {
                                    node {
                                      ref
                                      createdOn
                                      status
                                      type
                                      cardType
                                      paymentMethod
                                      total
                                      currency
                                    }
                                  }
                                }
                                attributes {
                                  name
                                  type
                                  value
                                }
                              }
                              comments(
                                first: $comments_first
                                entityId: [$id]
                                entityType: "ORDER"
                              ) {
                                edges {
                                  node {
                                    id
                                    text
                                    entityId
                                    entityType
                                    text
                                    createdOn
                                    updatedOn
                                    user {
                                      firstName
                                      lastName
                                    }
                                  }
                                }
                              }
                              returnOrders(
                                first: $returnOrders_first
                                order: { ref: $ref, retailer: { id: $retailerId } }
                              ) {
                                edges {
                                  node {
                                    id
                                    ref
                                    type
                                    status
                                    createdOn
                                    retailer {
                                      id
                                    }
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        id: '{{params.id}}',
                        ref: '{{params.ref}}',
                        retailerId: '{{params.retailerId}}',
                        fulfilments_first: 100,
                        items_first: 100,
                        comments_first: 100,
                        financialTransactions_first: 100,
                        returnOrders_first: 100,
                      },
                    },
                  props: {
                    title: '{{orderById.ref}} - {{orderById.status}}',
                    actions: {
                      primary: [
                        {
                          type: 'userAction',
                          name: 'ReturnOrder',
                          overrides: {
                            lodgedLocation: {
                              label:
                                'i18n:fc.om.orders.detail.userAction.returnOrder.lodgedLocation.label',
                            },
                            pickupLocation: {
                              label:
                                'i18n:fc.om.orders.detail.userAction.returnOrder.pickupLocation.label',
                            },
                          },
                        },
                      ],
                      secondary: [
                        {
                          type: 'mutation',
                          label: 'i18n:fc.om.orders.detail.userAction.addComment',
                          name: 'createComment',
                          filter: {
                            type: 'exclude',
                            names: ['entityRef'],
                          },
                          overrides: {
                            entityId: {
                              defaultValue: '{{orderById.id}}',
                            },
                            entityType: {
                              defaultValue: 'ORDER',
                            },
                          },
                        },
                        {
                          type: 'userAction',
                          name: 'OrderAddressChange',
                          overrides: {
                            deliveryAddress: {
                              defaultValue: {
                                companyName:
                                  '{{orderById.fulfilmentChoice.deliveryAddress.companyName}}',
                                name: '{{orderById.fulfilmentChoice.deliveryAddress.name}}',
                                street:
                                  '{{orderById.fulfilmentChoice.deliveryAddress.street}}',
                                city: '{{orderById.fulfilmentChoice.deliveryAddress.city}}',
                                postcode:
                                  '{{orderById.fulfilmentChoice.deliveryAddress.postcode}}',
                                state:
                                  '{{orderById.fulfilmentChoice.deliveryAddress.state}}',
                                country:
                                  '{{orderById.fulfilmentChoice.deliveryAddress.country}}',
                              },
                            },
                          },
                        },
                      ],
                    },
                    backButtons: [
                      {
                        path: 'orders',
                        menuLabel: 'i18n:fc.om.orders.detail.breadcrumb.backToOrders',
                      },
                      {
                        path: 'customers/{{orderById.customer.id}}/{{orderById.customer.ref}}',
                        menuLabel: 'i18n:fc.om.orders.detail.breadcrumb.viewCustomer',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.tabs',
                      props: {
                        layouts: [
                          {
                            label: 'i18n:fc.om.orders.detail.tab.details.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.fulfilments.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.transactions.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.attributes.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.returns.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.comments.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.activity.label',
                          },
                        ],
                      },
                      descendants: [
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title: 'i18n:fc.om.orders.detail.card.summary.title',
                                width: 'half',
                                dataSource: 'orderById',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.summary.attribute.orderType.label',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.summary.attribute.orderStatus.label',
                                    template: '{{status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.summary.attribute.creationDate.label',
                                    template:
                                      '{{dateStringFormatter createdOn}} ({{dateRelative createdOn}})',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.summary.attribute.updatedDate.label',
                                    template:
                                      '{{dateStringFormatter updatedOn}} ({{dateRelative updatedOn}})',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title:
                                  'i18n:fc.om.orders.detail.card.retailerInfo.title',
                                width: 'half',
                                dataSource: 'orderById',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.retailerInfo.attribute.retailerID.label',
                                    template: '{{retailer.id}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.retailerInfo.attribute.retailer.label',
                                    template: '{{retailer.tradingName}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.retailerInfo.attribute.commissionAmount.label',
                                    dataSource: 'attributes',
                                    filterFirst: {
                                      name: 'COMMISSION_AMOUNT',
                                      type: 'STRING',
                                    },
                                    template: '{{value}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title: 'i18n:fc.om.orders.detail.card.customer.title',
                                width: 'half',
                                dataSource: 'orderById.customer',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.customer.attribute.customerName.label',
                                    template: '{{firstName}} {{lastName}}',
                                    link: '#/customers/{{id}}/{{ref}}',
                                    condition: '{{and id ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.customer.attribute.phoneNo..label',
                                    template: '{{primaryPhone}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.customer.attribute.email.label',
                                    template: '{{primaryEmail}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.customer.attribute.timezone.label',
                                    template: '{{timezone}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.customer.attribute.id.label',
                                    template: '{{id}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.customer.attribute.ref.label',
                                    template: '{{ref}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title:
                                  'i18n:fc.om.orders.detail.card.deliveryInfo.title',
                                width: 'half',
                                dataSource: 'orderById.fulfilmentChoice',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.pickupLocationRef.label',
                                    template: '{{pickupLocationRef}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.customerName.label',
                                    template: '{{deliveryAddress.name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.street.label',
                                    template: '{{deliveryAddress.street}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.city.label',
                                    template: '{{deliveryAddress.city}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.state.label',
                                    template: '{{deliveryAddress.state}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.postcode.label',
                                    template: '{{deliveryAddress.postcode}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.country.label',
                                    template: '{{deliveryAddress.country}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.coordinates.label',
                                    template:
                                      '{{deliveryAddress.latitude}}, {{deliveryAddress.longitude}}',
                                    link_template:
                                      'https://maps.google.com/maps?q={{deliveryAddress.latitude}},{{deliveryAddress.longitude}}',
                                    condition:
                                      '{{and deliveryAddress.latitude deliveryAddress.longitude}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.deliveryType.label',
                                    template: '{{deliveryType}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.deliveryInstructions.label',
                                    template: '{{deliveryInstruction}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.card.deliveryInfo.attribute.fulfilmentPrice.label',
                                    template:
                                      '{{fulfilmentPrice}}{{#unless fulfilmentPrice}}0.00{{/unless}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.orders.detail.list.fulfilments.title',
                                dataSource: 'orderById.fulfilments',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.id.heading',
                                    template: '{{node.id}}',
                                    link_template: '#/fulfilment/{{node.id}}',
                                    condition: '{{and node.id}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.deliveryType.heading',
                                    template: '{{node.deliveryType}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.fulfilmentLocation.heading',
                                    template:
                                      '{{node.fromAddress.companyName}} {{#if node.fromAddress.name}}{{node.fromAddress.name}}{{/if}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.destination.heading',
                                    template:
                                      '{{node.toAddress.companyName}} {{node.toAddress.name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.createdOn.heading',
                                    template: '{{dateStringFormatter node.createdOn}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title: 'i18n:fc.om.orders.detail.list.orderItems.title',
                                dataSource: 'orderById.items',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orderItems.column.ref.heading',
                                    type: 'component',
                                    options: {
                                      component: 'fc.order.itemDetails',
                                    },
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orderItems.column.productName.heading',
                                    template: '{{node.product.name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orderItems.column.quantity.heading',
                                    template: '{{node.quantity}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orderItems.column.unitPrice.heading',
                                    template: '{{currency node.price node.currency}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orderItems.column.totalPrice.heading',
                                    template:
                                      '{{currency node.totalPrice node.currency}}',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.orders.detail.list.fulfilments.title',
                                dataSource: 'orderById.fulfilments',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.id.heading',
                                    template: '{{node.id}}',
                                    link_template: '#/fulfilment/{{node.id}}',
                                    condition: '{{and node.id}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.deliveryType.heading',
                                    template: '{{node.deliveryType}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.fulfilmentLocation.heading',
                                    template:
                                      '{{node.fromAddress.companyName}} {{#if node.fromAddress.name}}{{node.fromAddress.name}}{{/if}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.destination.heading',
                                    template:
                                      '{{node.toAddress.companyName}} {{node.toAddress.name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.fulfilments.column.createdOn.heading',
                                    template: '{{dateStringFormatter node.createdOn}}',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.orders.detail.list.transactions.title',
                                dataSource: 'orderById.financialTransactions',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.ref.heading',
                                    template: '{{node.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.createdOn.heading',
                                    template:
                                      "{{formatDate node.createdOn day='numeric' month='numeric' year='numeric'}}",
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.type.heading',
                                    template: '{{node.type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.cardType.heading',
                                    template: '{{node.cardType}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.paymentMethod.heading',
                                    template: '{{node.paymentMethod}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.transactions.column.amount.heading',
                                    template: '{{currency node.total node.currency}}',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title: 'i18n:fc.om.orders.detail.list.attributes.title',
                                dataSource: 'orderById.attributes',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.attributes.column.name.heading',
                                    template: '{{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.attributes.column.type.heading',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.attributes.column.value.heading',
                                    type: 'component',
                                    options: {
                                      component: 'fc.page.section',
                                      descendants: [
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: '{{type}}',
                                            matches: 'JSON',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.attribute.json',
                                              dataSource: 'value',
                                            },
                                          ],
                                        },
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: "{{eq type 'JSON'}}",
                                            matches: 'false',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.mystique.collapsible.text',
                                              props: {
                                                text: '{{value}}',
                                                charCutoff: 100,
                                              },
                                            },
                                          ],
                                        },
                                      ],
                                    },
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title: 'i18n:fc.om.orders.detail.list.returns.title',
                                defaultPageSize: 10,
                                dataSource: 'returnOrders',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orders.column.returnOrderRef.heading',
                                    template: '{{node.ref}}',
                                    link_template:
                                      '#/returns/{{node.id}}/{{node.retailer.id}}/{{node.ref}}',
                                    condition:
                                      '{{and node.id node.retailer.id node.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orders.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.orders.column.createdOn.heading',
                                    template:
                                      '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title: 'i18n:fc.om.orders.detail.list.comments.title',
                                dataSource: 'comments',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.comments.column.comment.heading',
                                    template: '{{node.text}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.comments.column.user.heading',
                                    template:
                                      '{{node.user.firstName}} {{node.user.lastName}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.comments.column.date.heading',
                                    template:
                                      '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.activity.entity',
                              dataSource: 'orderById',
                              props: {
                                width: '12',
                                showChildEntities: true,
                              },
                            },
                          ],
                        },
                      ],
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'orderitems/:id',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query ($id: ID!) {
                              orderItemById(id: $id) {
                                id
                                ref
                                price
                                status
                                quantity
                                createdOn
                                order {
                                  id
                                  ref
                                  retailer {
                                    id
                                  }
                                }
                                product {
                                  name
                                  ...vProduct
                                }
                              }
                            }
                            fragment vProduct on VariantProduct {
                              ref
                              attributes {
                                name
                                type
                                value
                              }
                            }
                          `,
                        ),
                      variables: {
                        id: '{{params.id}}',
                      },
                    },
                  props: {
                    title:
                      'Order {{orderItemById.order.ref}} Items - {{orderItemById.ref}}',
                    actions: true,
                    backButtons: [
                      {
                        path: 'orders/{{orderItemById.order.id}}/{{orderItemById.order.retailer.id}}/{{orderItemById.order.ref}}',
                        menuLabel: 'Back to Order ({{orderItemById.order.ref}})',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title: 'i18n:fc.om.orderitems.detail.card.summary.title',
                        width: 'half',
                        dataSource: 'orderItemById',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.card.summary.attribute.ref.label',
                            template: '{{ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.card.summary.attribute.status.label',
                            template: '{{status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.card.summary.attribute.quantity.label',
                            template: '{{quantity}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.card.summary.attribute.creationDate.label',
                            template:
                              '{{dateStringFormatter createdOn}} ({{dateRelative createdOn}})',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title: 'i18n:fc.om.orderitems.detail.card.product.title',
                        width: 'half',
                        dataSource: 'orderItemById.product',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.card.product.attribute.name.label',
                            template: '{{name}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.card.product.attribute.ref.label',
                            template: '{{ref}}',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.list',
                      props: {
                        title: 'i18n:fc.om.orderitems.detail.list.attributes.title',
                        dataSource: 'orderItemById.product.attributes',
                        responsiveness: 'card',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.list.attributes.column.name.heading',
                            template: '{{name}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.list.attributes.column.type.heading',
                            template: '{{type}}',
                          },
                          {
                            label:
                              'i18n:fc.om.orderitems.detail.list.attributes.column.value.heading',
                            type: 'component',
                            options: {
                              component: 'fc.page.section',
                              descendants: [
                                {
                                  component: 'fc.conditional',
                                  props: {
                                    value: '{{type}}',
                                    matches: 'JSON',
                                  },
                                  descendants: [
                                    {
                                      component: 'fc.attribute.json',
                                      dataSource: 'value',
                                    },
                                  ],
                                },
                                {
                                  component: 'fc.conditional',
                                  props: {
                                    value: "{{eq type 'JSON'}}",
                                    matches: 'false',
                                  },
                                  descendants: [
                                    {
                                      component: 'fc.mystique.collapsible.text',
                                      props: {
                                        text: '{{value}}',
                                        charCutoff: 100,
                                      },
                                    },
                                  ],
                                },
                              ],
                            },
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  path: 'escalatedFulfilments',
                  type: 'page',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query ($ref: [String], $fulfilments_first: Int) {
                              fulfilments(
                                first: $fulfilments_first
                                ref: $ref
                                status: ["ESCALATED"]
                              ) {
                                edges {
                                  node {
                                    id
                                    ref
                                    createdOn
                                    order {
                                      ref
                                      id
                                      status
                                      retailer {
                                        id
                                      }
                                      customer {
                                        firstName
                                        lastName
                                      }
                                    }
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        fulfilments_first: 100,
                      },
                    },
                  nav: {
                    label: 'i18n:fc.om.fulfilment.index.title',
                    icon: 'warning',
                  },
                  props: {
                    title: 'i18n:fc.om.fulfilment.index.title',
                  },
                  descendants: [
                    {
                      component: 'fc.list',
                      props: {
                        defaultPageSize: 10,
                        dataSource: 'fulfilments',
                        responsiveness: 'card',
                        filter: {
                          enabled: true,
                          exclude: [
                            'deliveryType',
                            'eta',
                            'type',
                            'workflowRef',
                            'workflowVersion',
                          ],
                        },
                        attributes: [
                          {
                            label: 'i18n:fc.om.fulfilment.index.list.column.id.heading',
                            template: '{{node.id}}',
                            link_template: '#/fulfilment/{{node.id}}',
                            condition: '{{and node.id}}',
                          },
                          {
                            label:
                              'i18n:fc.om.fulfilment.index.list.column.orderRef.heading',
                            template: '{{node.order.ref}}',
                            link_template:
                              '#/orders/{{node.order.id}}/{{node.order.retailer.id}}/{{node.order.ref}}',
                            condition:
                              '{{and node.order.id node.order.retailer.id node.order.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.fulfilment.index.list.column.customer.heading',
                            template:
                              '{{node.order.customer.firstName}} {{node.order.customer.lastName}}',
                          },
                          {
                            label:
                              'i18n:fc.om.fulfilment.index.list.column.orderStatus.heading',
                            template: '{{node.order.status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.fulfilment.index.list.column.creationDate.heading',
                            template:
                              '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  path: 'fulfilment/:id',
                  type: 'page',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $id: ID!
                              $items_after: String
                              $items_first: Int
                              $articles_after: String
                              $articles_first: Int
                            ) {
                              fulfilmentById(id: $id) {
                                id
                                status
                                ref
                                eta
                                createdOn
                                expiryTime
                                deliveryType
                                order {
                                  id
                                  ref
                                  retailer {
                                    id
                                  }
                                }
                                fromAddress {
                                  ref
                                  companyName
                                  name
                                  street
                                  city
                                  state
                                  postcode
                                  country
                                }
                                toAddress {
                                  ref
                                  companyName
                                  name
                                  street
                                  city
                                  state
                                  postcode
                                  country
                                }
                                articles(after: $articles_after, first: $articles_first) {
                                  edges {
                                    node {
                                      id
                                      ref
                                      status
                                      createdOn
                                    }
                                  }
                                }
                                items(after: $items_after, first: $items_first) {
                                  edges {
                                    node {
                                      orderItem {
                                        ref
                                        product {
                                          name
                                        }
                                      }
                                      requestedQuantity
                                      filledQuantity
                                      rejectedQuantity
                                    }
                                  }
                                }
                                attributes {
                                  name
                                  type
                                  value
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        id: '{{params.id}}',
                        items_first: 100,
                        articles_first: 100,
                      },
                    },
                  props: {
                    title:
                      'Fulfilment - {{fulfilmentById.id}} - {{fulfilmentById.status}}',
                    actions: true,
                    backButtons: [
                      {
                        path: 'orders/{{fulfilmentById.order.id}}/{{fulfilmentById.order.retailer.id}}/{{fulfilmentById.order.ref}}',
                        menuLabel: 'Back to Order ({{fulfilmentById.order.ref}})',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.page.section.header',
                      descendants: [
                        {
                          component: 'fc.button.print.download',
                          props: {
                            label: 'i18n:fc.om.fulfilment.detail.button.print.label',
                            href: 'api/v4/fulfilment/inpack/{{fulfilmentById.id}}',
                            filename: 'Packing_Slip_{{fulfilmentById.id}}.pdf',
                          },
                        },
                      ],
                    },
                    {
                      component: 'fc.tabs',
                      props: {
                        layouts: [
                          {
                            label: 'i18n:fc.om.fulfilment.detail.tab.details.label',
                          },
                          {
                            label: 'i18n:fc.om.fulfilment.detail.tab.attributes.label',
                          },
                        ],
                      },
                      descendants: [
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title:
                                  'i18n:fc.om.fulfilment.detail.card.overview.title',
                                dataSource: 'fulfilmentById',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.overview.attribute.iD.label',
                                    template: '{{id}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.overview.attribute.ref.label',
                                    template: '{{ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.overview.attribute.status.label',
                                    template: '{{status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.overview.attribute.deliveryType.label',
                                    template: '{{deliveryType}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.overview.attribute.creationDate.label',
                                    template:
                                      '{{dateStringFormatter createdOn}} ({{dateRelative createdOn}})',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.overview.attribute.eTA.label',
                                    template:
                                      '{{#if expiryTime}}{{dateStringFormatter expiryTime}} ({{dateRelative expiryTime}}){{/if}}{{#unless expiryTime}}No Expiry Date Set{{/unless}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title:
                                  'i18n:fc.om.fulfilment.detail.card.fromLocation.title',
                                width: 'half',
                                dataSource: 'fulfilmentById.fromAddress',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.fromLocation.attribute.from.label',
                                    template: '{{ref}} {{companyName}} {{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.fromLocation.attribute.street.label',
                                    template: '{{street}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.fromLocation.attribute.city.label',
                                    template: '{{city}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.fromLocation.attribute.state.label',
                                    template: '{{state}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.fromLocation.attribute.postcode.label',
                                    template: '{{postcode}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.fromLocation.attribute.country.label',
                                    template: '{{country}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title:
                                  'i18n:fc.om.fulfilment.detail.card.toLocation.title',
                                width: 'half',
                                dataSource: 'fulfilmentById.toAddress',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.toLocation.attribute.to.label',
                                    template: '{{ref}} {{companyName}} {{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.toLocation.attribute.street.label',
                                    template: '{{street}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.toLocation.attribute.city.label',
                                    template: '{{city}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.toLocation.attribute.state.label',
                                    template: '{{state}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.toLocation.attribute.postcode.label',
                                    template: '{{postcode}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.card.toLocation.attribute.country.label',
                                    template: '{{country}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.fulfilment.detail.list.fulfilmentItems.title',
                                dataSource: 'fulfilmentById.items',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.fulfilmentItems.column.productName.heading',
                                    template: '{{node.orderItem.product.name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.fulfilmentItems.column.skuRef.heading',
                                    template: '{{node.orderItem.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.fulfilmentItems.column.requestedQuantity.heading',
                                    template: '{{node.requestedQuantity}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.fulfilmentItems.column.filledQuantity.heading',
                                    template: '{{node.filledQuantity}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.fulfilmentItems.column.rejectedQuantity.heading',
                                    template: '{{node.rejectedQuantity}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.fulfilment.detail.list.articles.title',
                                dataSource: 'fulfilmentById.articles',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.articles.column.articleRef.heading',
                                    template: '{{node.ref}}',
                                    link_template: '#/article/{{node.id}}',
                                    condition: '{{and node.id}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.articles.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.articles.column.creationDate.heading',
                                    template:
                                      '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.fulfilment.detail.tab.attributes.label',
                                dataSource: 'fulfilmentById.attributes',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.attributes.column.name.heading',
                                    template: '{{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.attributes.column.type.heading',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.fulfilment.detail.list.attributes.column.value.heading',
                                    type: 'component',
                                    options: {
                                      component: 'fc.page.section',
                                      descendants: [
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: '{{type}}',
                                            matches: 'JSON',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.attribute.json',
                                              dataSource: 'value',
                                            },
                                          ],
                                        },
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: "{{eq type 'JSON'}}",
                                            matches: 'false',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.mystique.collapsible.text',
                                              props: {
                                                text: '{{value}}',
                                                charCutoff: 100,
                                              },
                                            },
                                          ],
                                        },
                                      ],
                                    },
                                  },
                                ],
                              },
                            },
                          ],
                        },
                      ],
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'customers',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query ($customers_after: String, $customers_first: Int) {
                              customers(after: $customers_after, first: $customers_first) {
                                pageInfo {
                                  hasNextPage
                                  hasPreviousPage
                                }
                                edges {
                                  cursor
                                  node {
                                    id
                                    ref
                                    firstName
                                    lastName
                                    primaryEmail
                                    primaryPhone
                                    timezone
                                    status
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        customers_first: 100,
                      },
                    },
                  nav: {
                    label: 'i18n:fc.om.customers.index.nav',
                    icon: 'face',
                  },
                  props: {
                    title: 'i18n:fc.om.customers.index.title',
                  },
                  descendants: [
                    {
                      component: 'fc.list',
                      props: {
                        dataSource: 'customers',
                        responsiveness: 'card',
                        filter: {
                          enabled: true,
                          exclude: [
                            'workflowRef',
                            'workflowVersion',
                            'title',
                            'department',
                            'country',
                            'username',
                            'timezone',
                          ],
                        },
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.customers.index.list.column.name.heading',
                            template: '{{node.firstName}} {{node.lastName}}',
                            link_template: '#/customers/{{node.id}}/{{node.ref}}',
                            condition: '{{and node.id node.ref}}',
                          },
                          {
                            label: 'i18n:fc.om.customers.index.list.column.ref.heading',
                            template: '{{node.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.index.list.column.email.heading',
                            template: '{{node.primaryEmail}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.index.list.column.phone.heading',
                            template: '{{node.primaryPhone}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.index.list.column.timezone.heading',
                            template: '{{node.timezone}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.index.list.column.status.heading',
                            template: '{{node.status}}',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'customers/:id/:ref',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $id: ID!
                              $ref: String!
                              $billingAccounts_first: Int
                              $orders_first: Int
                            ) {
                              customerById(id: $id) {
                                id
                                ref
                                firstName
                                lastName
                                username
                                timezone
                                primaryEmail
                                primaryPhone
                                status
                                retailer {
                                  id
                                  tradingName
                                }
                                country
                                department
                                createdOn
                                updatedOn
                              }
                              orders(first: $orders_first, customerLink: { ref: $ref }) {
                                edges {
                                  node {
                                    id
                                    ref
                                    retailer {
                                      id
                                      tradingName
                                    }
                                    type
                                    status
                                    retailer {
                                      id
                                    }
                                    workflowRef
                                    workflowVersion
                                    totalPrice
                                    totalTaxPrice
                                    createdOn
                                    customer {
                                      id
                                      firstName
                                      lastName
                                    }
                                  }
                                }
                              }
                              billingAccounts(
                                first: $billingAccounts_first
                                customer: { ref: $ref }
                              ) {
                                edges {
                                  node {
                                    id
                                    ref
                                    type
                                    status
                                    retailer {
                                      id
                                    }
                                    workflowRef
                                    workflowVersion
                                    createdOn
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        id: '{{params.id}}',
                        ref: '{{params.ref}}',
                        billingAccounts_first: 100,
                        orders_first: 100,
                      },
                    },
                  props: {
                    title:
                      'Customer Details - {{customerById.firstName}} {{customerById.lastName}}',
                    backButtons: [
                      {
                        path: 'customers',
                        menuLabel:
                          'i18n:fc.om.customers.detail.breadcrumb.backToCustomers',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title:
                          'i18n:fc.om.customers.detail.card.personalInformation.title',
                        width: 'half',
                        dataSource: 'customerById',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.personalInformation.attribute.name.label',
                            template: '{{firstName}} {{lastName}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.personalInformation.attribute.primaryPhone.label',
                            template: '{{primaryPhone}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.personalInformation.attribute.primaryEmail.label',
                            template: '{{primaryEmail}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.personalInformation.attribute.country.label',
                            template: '{{country}}',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title: 'i18n:fc.om.customers.detail.card.accountDetails.title',
                        width: 'half',
                        dataSource: 'customerById',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.accountDetails.attribute.id.label',
                            template: '{{id}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.accountDetails.attribute.ref.label',
                            template: '{{ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.accountDetails.attribute.status.label',
                            template: '{{status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.accountDetails.attribute.createdOn.label',
                            template:
                              '{{dateStringFormatter createdOn}} ({{dateRelative createdOn}})',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.card.accountDetails.attribute.updatedOn.label',
                            template:
                              '{{dateStringFormatter updatedOn}} ({{dateRelative updatedOn}})',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.list',
                      props: {
                        title: 'i18n:fc.om.customers.detail.list.billingAccounts.title',
                        defaultPageSize: 10,
                        dataSource: 'billingAccounts',
                        responsiveness: 'card',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.billingAccounts.column.ref.heading',
                            template: '{{node.ref}}',
                            link_template: '#/billing/{{node.ref}}',
                            condition: '{{and node.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.billingAccounts.column.status.heading',
                            template: '{{node.status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.billingAccounts.column.type.heading',
                            template: '{{node.type}}',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.list',
                      props: {
                        title: 'i18n:fc.om.customers.detail.list.orderHistory.title',
                        defaultPageSize: 10,
                        dataSource: 'orders',
                        responsiveness: 'card',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.orderHistory.column.orderRef.heading',
                            template: '{{node.ref}}',
                            link_template:
                              '#/orders/{{node.id}}/{{node.retailer.id}}/{{node.ref}}',
                            condition: '{{and node.id node.retailer.id node.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.orderHistory.column.status.heading',
                            template: '{{node.status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.orderHistory.column.orderType.heading',
                            template: '{{node.type}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.orderHistory.column.orderValue.heading',
                            template: '{{node.totalPrice}}',
                          },
                          {
                            label:
                              'i18n:fc.om.customers.detail.list.orderHistory.column.creationDate.heading',
                            template: '{{dateRelative node.createdOn}}',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'returns',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query ($returnOrders_first: Int) {
                              returnOrders(first: $returnOrders_first) {
                                edges {
                                  node {
                                    id
                                    retailer {
                                      id
                                    }
                                    ref
                                    order {
                                      ref
                                    }
                                    type
                                    status
                                    createdOn
                                    returnAuthorisationKey
                                    returnAuthorisationKeyExpiry
                                    workflowRef
                                    lodgedLocation {
                                      ref
                                    }
                                    destinationLocation {
                                      ref
                                    }
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        returnOrders_first: 100,
                      },
                    },
                  nav: {
                    label: 'i18n:fc.om.returns.index.nav',
                    icon: 'call_missed',
                  },
                  props: {
                    title: 'i18n:fc.om.returns.index.title',
                  },
                  descendants: [
                    {
                      component: 'fc.list',
                      props: {
                        defaultPageSize: 10,
                        dataSource: 'returnOrders',
                        responsiveness: 'card',
                        filter: {
                          enabled: true,
                          exclude: [
                            'workflowRef',
                            'workflowVersion',
                            'returnAuthorisationKey',
                          ],
                        },
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.returns.index.list.column.returnOrderRef.heading',
                            template: '{{node.ref}}',
                            link_template:
                              '#/returns/{{node.id}}/{{node.retailer.id}}/{{node.ref}}',
                            condition: '{{and node.id node.retailer.id node.ref}}',
                          },
                          {
                            label: 'i18n:fc.om.returns.index.list.column.type.heading',
                            template: '{{node.type}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returns.index.list.column.orderRef.heading',
                            template: '{{node.order.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returns.index.list.column.status.heading',
                            template: '{{node.status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returns.index.list.column.lodgedLocation.heading',
                            template: '{{node.lodgedLocation.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returns.index.list.column.destinationLocation.heading',
                            template: '{{node.destinationLocation.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returns.index.list.column.creationDate.heading',
                            template:
                              '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'returns/:id/:retailerId/:ref',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $ref: String!
                              $retailerId: ID!
                              $returnOrderFulfilments_first: Int
                              $returnOrderItems_first: Int
                            ) {
                              returnOrder(ref: $ref, retailer: { id: $retailerId }) {
                                id
                                __typename
                                ref
                                status
                                retailer {
                                  id
                                }
                                type
                                customer {
                                  ref
                                }
                                order {
                                  ref
                                }
                                exchangeOrder {
                                  ref
                                }
                                creditMemo {
                                  ref
                                }
                                returnOrderItems(first: $returnOrderItems_first) {
                                  edges {
                                    node {
                                      ref
                                      status
                                      orderItem {
                                        ref
                                      }
                                      product {
                                        ref
                                      }
                                      returnReason {
                                        value
                                        label
                                      }
                                      returnReasonComment
                                      returnCondition {
                                        value
                                        label
                                      }
                                    }
                                  }
                                }
                                returnVerifications {
                                  edges {
                                    node {
                                      id
                                    }
                                  }
                                }
                                returnAuthorisationKey
                                lodgedLocation {
                                  ref
                                }
                                destinationLocation {
                                  ref
                                }
                                pickupAddress {
                                  companyName
                                  name
                                  street
                                  city
                                  state
                                  postcode
                                  region
                                  country
                                  latitude
                                  longitude
                                  timeZone
                                }
                                destinationLocation {
                                  ref
                                }
                                createdOn
                                updatedOn
                                workflow {
                                  ref
                                  version
                                }
                                returnOrderFulfilments(
                                  first: $returnOrderFulfilments_first
                                ) {
                                  edges {
                                    node {
                                      ref
                                      type
                                      returnOrder {
                                        ref
                                        retailer {
                                          id
                                        }
                                      }
                                      lodgedLocation {
                                        ref
                                      }
                                      destinationLocation {
                                        ref
                                      }
                                      pickupAddress {
                                        name
                                      }
                                      status
                                      workflowRef
                                    }
                                  }
                                }
                                attributes {
                                  name
                                  type
                                  value
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        ref: '{{params.ref}}',
                        retailerId: '{{params.retailerId}}',
                        returnOrderFulfilments_first: 100,
                        returnOrderItems_first: 100,
                      },
                    },
                  props: {
                    title: '{{returnOrder.ref}} - {{returnOrder.status}}',
                    actions: true,
                    backButtons: [
                      {
                        path: 'returns',
                        menuLabel:
                          'i18n:fc.om.returns.detail.breadcrumb.backToReturnOrders',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.tabs',
                      props: {
                        layouts: [
                          {
                            label: 'i18n:fc.om.returns.detail.tab.details.label',
                          },
                          {
                            label: 'i18n:fc.om.returns.detail.tab.attributes.label',
                          },
                          {
                            label: 'i18n:fc.om.returns.detail.tab.activity.label',
                          },
                        ],
                      },
                      descendants: [
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title: 'i18n:fc.om.returns.detail.card.summary.title',
                                width: 'half',
                                dataSource: 'returnOrder',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.returnOrderType.label',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.returnOrderStatus.label',
                                    template: '{{status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.relatedOrder.label',
                                    template: '{{order.ref}}',
                                    link_template:
                                      '#/orders?orders_ref[]={{order.ref}}',
                                    condition: '{{and order.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.exchangeOrder.label',
                                    template: '{{exchangeOrder.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.crediMemo.label',
                                    template: '{{creditMemo.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.createdOn.label',
                                    template:
                                      '{{dateStringFormatter createdOn}} ({{dateRelative createdOn}})',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.summary.attribute.updatedOn.label',
                                    template:
                                      '{{dateStringFormatter updatedOn}} ({{dateRelative updatedOn}})',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title: 'i18n:fc.om.returns.detail.card.addresses.title',
                                width: 'half',
                                dataSource: 'returnOrder',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.lodgedLocation.label',
                                    template: '{{lodgedLocation.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.destinationLocation.label',
                                    template: '{{destinationLocation.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.pickupAddressName.label',
                                    template: '{{pickupAddress.name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.companyName.label',
                                    template: '{{pickupAddress.companyName}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.street.label',
                                    template: '{{pickupAddress.street}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.city.label',
                                    template: '{{pickupAddress.city}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.state/Postcode.label',
                                    template:
                                      '{{pickupAddress.state}} {{pickupAddress.postcode}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.region.label',
                                    template: '{{pickupAddress.region}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.card.addresses.attribute.country.label',
                                    template: '{{pickupAddress.country}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.returns.detail.list.returnFulfilments.title',
                                dataSource: 'returnOrder.returnOrderFulfilments',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnFulfilments.column.ref.heading',
                                    template: '{{node.ref}}',
                                    link_template:
                                      '#/returnFulfilment/{{node.returnOrder.ref}}/{{node.returnOrder.retailer.id}}/{{node.ref}}',
                                    condition:
                                      '{{and node.returnOrder.ref node.returnOrder.retailer.id node.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnFulfilments.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnFulfilments.column.lodgedLocation.heading',
                                    template:
                                      '{{#if node.lodgedLocation}}{{node.lodgedLocation.ref}}{{/if}}{{#if node.pickupAddress}}{{node.pickupAddress.name}}{{/if}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnFulfilments.column.destinationLocation.heading',
                                    template: '{{node.destinationLocation.ref}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.returns.detail.list.returnOrderItems.title',
                                dataSource: 'returnOrder.returnOrderItems',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.returnItemRef.heading',
                                    template: '{{node.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.orderItemRef.heading',
                                    template: '{{node.orderItem.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.productRef.heading',
                                    template: '{{node.product.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.reason.heading',
                                    template: '{{node.returnReason.label}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.condition.heading',
                                    template: '{{node.returnCondition.label}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.detail.list.returnOrderItems.column.comment.heading',
                                    template: '{{node.returnReasonComment}}',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.returns.attributes.list.attributes.title',
                                dataSource: 'returnOrder.attributes',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.returns.attributes.list.attributes.column.name.heading',
                                    template: '{{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.attributes.list.attributes.column.type.heading',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.returns.attributes.list.attributes.column.value.heading',
                                    type: 'component',
                                    options: {
                                      component: 'fc.page.section',
                                      descendants: [
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: '{{type}}',
                                            matches: 'JSON',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.attribute.json',
                                              dataSource: 'value',
                                            },
                                          ],
                                        },
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: "{{eq type 'JSON'}}",
                                            matches: 'false',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.mystique.collapsible.text',
                                              props: {
                                                text: '{{value}}',
                                                charCutoff: 100,
                                              },
                                            },
                                          ],
                                        },
                                      ],
                                    },
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.activity.entity',
                              dataSource: 'returnOrder',
                              props: {
                                width: '12',
                                showChildEntities: true,
                              },
                            },
                          ],
                        },
                      ],
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'returnFulfilment/:returnOrderRef/:retailerId/:ref',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $ref: String!
                              $returnOrderRef: String!
                              $retailerId: ID!
                            ) {
                              returnFulfilment(
                                ref: $ref
                                returnOrder: {
                                  ref: $returnOrderRef
                                  retailer: { id: $retailerId }
                                }
                              ) {
                                id
                                status
                                ref
                                createdOn
                                returnOrder {
                                  id
                                  ref
                                  retailer {
                                    id
                                  }
                                }
                                lodgedLocation {
                                  ref
                                }
                                pickupAddress {
                                  companyName
                                  name
                                  street
                                  city
                                  state
                                  postcode
                                  region
                                  country
                                  latitude
                                  longitude
                                  timeZone
                                }
                                destinationLocation {
                                  ref
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        ref: '{{params.ref}}',
                        returnOrderRef: '{{params.returnOrderRef}}',
                        retailerId: '{{params.retailerId}}',
                      },
                    },
                  props: {
                    title: 'Return Fulfilment - {{returnFulfilment.status}}',
                    actions: true,
                    backButtons: [
                      {
                        path: 'returns/{{returnFulfilment.returnOrder.id}}/{{returnFulfilment.returnOrder.retailer.id}}/{{returnFulfilment.returnOrder.ref}}',
                        menuLabel:
                          'Back to Return Order ({{returnFulfilment.returnOrder.ref}})',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title: 'i18n:fc.om.returnFulfilment.detail.card.overview.title',
                        dataSource: 'returnFulfilment',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.overview.attribute.ID.label',
                            template: '{{id}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.overview.attribute.ref.label',
                            template: '{{ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.overview.attribute.status.label',
                            template: '{{status}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.overview.attribute.creationDate.label',
                            template:
                              '{{dateStringFormatter createdOn}} ({{dateRelative createdOn}})',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title:
                          'i18n:fc.om.returnFulfilment.detail.card.fromLocation.title',
                        width: 'half',
                        dataSource: 'returnFulfilment',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.lodgedLocationRef.label',
                            template: '{{lodgedLocation.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.pickupAddressName.label',
                            template: '{{pickupAddress.name}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.companyName.label',
                            template: '{{pickupAddress.companyName}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.street.label',
                            template: '{{pickupAddress.street}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.city.label',
                            template: '{{pickupAddress.city}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.statePostcode.label',
                            template:
                              '{{pickupAddress.state}} {{pickupAddress.postcode}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.region.label',
                            template: '{{pickupAddress.region}}',
                          },
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.fromLocation.attribute.country.label',
                            template: '{{pickupAddress.country}}',
                          },
                        ],
                      },
                    },
                    {
                      component: 'fc.card.attribute',
                      props: {
                        title:
                          'i18n:fc.om.returnFulfilment.detail.card.toLocation.title',
                        width: 'half',
                        dataSource: 'returnFulfilment',
                        attributes: [
                          {
                            label:
                              'i18n:fc.om.returnFulfilment.detail.card.toLocation.attribute.destinationLocation.label',
                            template: '{{destinationLocation.ref}}',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  type: 'page',
                  path: 'billing',
                  component: 'fc.page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $billingAccounts_first: Int
                              $billingAccounts_after: String
                            ) {
                              billingAccounts(
                                first: $billingAccounts_first
                                after: $billingAccounts_after
                              ) {
                                edges {
                                  node {
                                    id
                                    retailer {
                                      id
                                    }
                                    ref
                                    type
                                    status
                                    retailer {
                                      id
                                    }
                                    customer {
                                      ref
                                    }
                                    name
                                    createdOn
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        billingAccounts_first: 100,
                      },
                    },
                  nav: {
                    label: 'i18n:fc.om.billing.index.nav',
                    icon: 'local_atm',
                  },
                  props: {
                    title: 'i18n:fc.om.billing.index.title',
                  },
                  descendants: [
                    {
                      component: 'fc.list',
                      props: {
                        defaultPageSize: 10,
                        dataSource: 'billingAccounts',
                        responsiveness: 'card',
                        filter: {
                          enabled: true,
                          exclude: ['workflowRef', 'workflowVersion'],
                        },
                        attributes: [
                          {
                            label: 'i18n:fc.om.billing.index.list.column.name.heading',
                            template: '{{node.name}}',
                            link: '#/billing/{{node.ref}}',
                            condition: '{{and node.ref}}',
                          },
                          {
                            label:
                              'i18n:fc.om.billing.index.list.column.status.heading',
                            template: '{{node.status}}',
                          },
                          {
                            label: 'i18n:fc.om.billing.index.list.column.type.heading',
                            template: '{{node.type}}',
                          },
                          {
                            label:
                              'i18n:fc.om.billing.index.list.column.customerRef.heading',
                            template: '{{node.customer.ref}}',
                          },
                        ],
                      },
                    },
                  ],
                },
                {
                  path: 'billing/:ref',
                  component: 'fc.page',
                  type: 'page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $ref: String!
                              $creditMemos_first: Int
                              $creditMemos_after: String
                            ) {
                              billingAccount(ref: $ref) {
                                id
                                __typename
                                ref
                                type
                                status
                                retailer {
                                  id
                                }
                                customer {
                                  ref
                                }
                                name
                                createdOn
                                creditMemos(
                                  first: $creditMemos_first
                                  after: $creditMemos_after
                                ) {
                                  edges {
                                    node {
                                      ref
                                      type
                                      status
                                      order {
                                        ref
                                      }
                                      issueDate
                                      totalAmount {
                                        amount
                                      }
                                      currency {
                                        alphabeticCode
                                      }
                                      createdOn
                                      workflowRef
                                    }
                                  }
                                }
                                attributes {
                                  name
                                  type
                                  value
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        ref: '{{params.ref}}',
                        creditMemos_first: 100,
                      },
                    },
                  props: {
                    title: 'Billing Account - {{billingAccount.name}}',
                    actions: true,
                    backButtons: [
                      {
                        path: 'billing',
                        menuLabel:
                          'i18n:fc.om.billing.detail.breadcrumb.backToAccountList',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.tabs',
                      props: {
                        layouts: [
                          {
                            label: 'i18n:fc.om.billing.detail.tab.details.label',
                          },
                          {
                            label: 'i18n:fc.om.billing.detail.tab.attributes.label',
                          },
                          {
                            label: 'i18n:fc.om.billing.detail.tab.activity.label',
                          },
                        ],
                      },
                      descendants: [
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title: 'i18n:fc.om.billing.detail.card.summary.title',
                                width: 'half',
                                dataSource: 'billingAccount',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.card.summary.attribute.billingAccountType.label',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.card.summary.attribute.billingAccountStatus.label',
                                    template: '{{status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.card.summary.attribute.billingAccountName.label',
                                    template: '{{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.card.summary.attribute.customerRef.label',
                                    template: '{{customer.ref}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.billing.detail.list.creditMemos.title',
                                dataSource: 'billingAccount.creditMemos',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.creditMemos.column.ref.heading',
                                    template: '{{node.ref}}',
                                    link: '#/creditmemo/{{node.ref}}',
                                    condition: '{{and node.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.creditMemos.column.amount.heading',
                                    template:
                                      '{{currency node.totalAmount.amount node.currency.alphabeticCode}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.creditMemos.column.status.heading',
                                    template: '{{node.status}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.creditMemos.column.type.heading',
                                    template: '{{node.type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.creditMemos.column.order.heading',
                                    template: '{{node.order.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.creditMemos.column.issueDate.heading',
                                    template:
                                      '{{dateStringFormatter node.issueDate}} ({{dateRelative node.issueDate}})',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.billing.detail.list.attributes.title',
                                dataSource: 'billingAccount.attributes',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.attributes.column.name.heading',
                                    template: '{{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.attributes.column.type.heading',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.billing.detail.list.attributes.column.value.heading',
                                    type: 'component',
                                    options: {
                                      component: 'fc.page.section',
                                      descendants: [
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: '{{type}}',
                                            matches: 'JSON',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.attribute.json',
                                              dataSource: 'value',
                                            },
                                          ],
                                        },
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: "{{eq type 'JSON'}}",
                                            matches: 'false',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.mystique.collapsible.text',
                                              props: {
                                                text: '{{value}}',
                                                charCutoff: 100,
                                              },
                                            },
                                          ],
                                        },
                                      ],
                                    },
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.activity.entity',
                              dataSource: 'billingAccount',
                              props: {
                                width: '12',
                                showChildEntities: false,
                              },
                            },
                          ],
                        },
                      ],
                    },
                  ],
                },
                {
                  path: 'creditmemo/:ref',
                  component: 'fc.page',
                  type: 'page',
                    data: {
                        query: parse(
                          gql`
                            query (
                              $ref: String!
                              $items_first: Int
                              $comments_first: Int
                              $items_after: String
                            ) {
                              creditMemo(ref: $ref) {
                                id
                                __typename
                                ref
                                type
                                status
                                billingAccount {
                                  ref
                                  name
                                }
                                items(first: $items_first, after: $items_after) {
                                  edges {
                                    node {
                                      ref
                                      type
                                      description
                                      createdOn
                                    }
                                  }
                                }
                                attributes {
                                  name
                                  type
                                  value
                                }
                                createdOn
                              }
                              comments(
                                first: $comments_first
                                entityRef: [$ref]
                                entityType: "CREDIT_MEMO"
                              ) {
                                edges {
                                  node {
                                    text
                                    createdOn
                                    user {
                                      firstName
                                      lastName
                                    }
                                  }
                                }
                              }
                            }
                          `,
                        ),
                      variables: {
                        ref: '{{params.ref}}',
                        comments_first: 100,
                        items_first: 100,
                      },
                    },
                  props: {
                    title: 'Credit Memo - {{creditMemo.ref}}',
                    actions: {
                      primary: [
                        {
                          type: 'mutation',
                          label: 'i18n:fc.om.orders.detail.userAction.addComment',
                          name: 'createComment',
                          filter: {
                            type: 'exclude',
                            names: ['entityId'],
                          },
                          overrides: {
                            entityRef: {
                              defaultValue: '{{creditMemo.ref}}',
                            },
                            entityType: {
                              defaultValue: 'CREDIT_MEMO',
                            },
                          },
                        },
                      ],
                    },
                    backButtons: [
                      {
                        path: 'billing/{{creditMemo.billingAccount.ref}}',
                        menuLabel:
                          'Back to Billing Account ({{creditMemo.billingAccount.name}})',
                      },
                    ],
                  },
                  descendants: [
                    {
                      component: 'fc.tabs',
                      props: {
                        layouts: [
                          {
                            label: 'i18n:fc.om.creditmemo.detail.tab.details.label',
                          },
                          {
                            label: 'i18n:fc.om.creditmemo.detail.tab.attributes.label',
                          },
                          {
                            label: 'i18n:fc.om.creditmemo.detail.tab.activity.label',
                          },
                          {
                            label: 'i18n:fc.om.orders.detail.tab.comments.label',
                          },
                        ],
                      },
                      descendants: [
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.card.attribute',
                              props: {
                                title:
                                  'i18n:fc.om.creditmemo.detail.card.summary.title',
                                width: 'half',
                                dataSource: 'creditMemo',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.card.summary.attribute.creditMemoType.label',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.card.summary.attribute.creditMemoStatus.label',
                                    template: '{{status}}',
                                  },
                                ],
                              },
                            },
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.creditmemo.detail.list.creditMemoItems.title',
                                dataSource: 'creditMemo.items',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.creditMemoItems.column.ref.heading',
                                    template: '{{node.ref}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.creditMemoItems.column.type.heading',
                                    template: '{{node.type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.creditMemoItems.column.description.heading',
                                    template: '{{node.description}}',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title:
                                  'i18n:fc.om.creditmemo.detail.list.attributes.title',
                                dataSource: 'creditMemo.attributes',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.attributes.column.name.heading',
                                    template: '{{name}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.attributes.column.type.heading',
                                    template: '{{type}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.creditmemo.detail.list.attributes.column.value.heading',
                                    type: 'component',
                                    options: {
                                      component: 'fc.page.section',
                                      descendants: [
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: '{{type}}',
                                            matches: 'JSON',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.attribute.json',
                                              dataSource: 'value',
                                            },
                                          ],
                                        },
                                        {
                                          component: 'fc.conditional',
                                          props: {
                                            value: "{{eq type 'JSON'}}",
                                            matches: 'false',
                                          },
                                          descendants: [
                                            {
                                              component: 'fc.mystique.collapsible.text',
                                              props: {
                                                text: '{{value}}',
                                                charCutoff: 100,
                                              },
                                            },
                                          ],
                                        },
                                      ],
                                    },
                                  },
                                ],
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.activity.entity',
                              dataSource: 'creditMemo',
                              props: {
                                width: '12',
                                showChildEntities: true,
                              },
                            },
                          ],
                        },
                        {
                          component: 'fc.page.section',
                          descendants: [
                            {
                              component: 'fc.list',
                              props: {
                                title: 'i18n:fc.om.orders.detail.list.comments.title',
                                dataSource: 'comments',
                                responsiveness: 'card',
                                attributes: [
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.comments.column.comment.heading',
                                    template: '{{node.text}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.comments.column.user.heading',
                                    template:
                                      '{{node.user.firstName}} {{node.user.lastName}}',
                                  },
                                  {
                                    label:
                                      'i18n:fc.om.orders.detail.list.comments.column.date.heading',
                                    template:
                                      '{{dateStringFormatter node.createdOn}} ({{dateRelative node.createdOn}})',
                                  },
                                ],
                              },
                            },
                          ],
                        },
                      ],
                    },
                  ],
                },
            ],
        },
    ],
};
