import { Loading } from 'mystique/components/Loading';
import {
  createStyles,
  IconButton,
  makeStyles,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
} from '@material-ui/core';
import { GoPlus } from 'react-icons/go';
import { ChevronLeft, ChevronRight } from '@material-ui/icons';
import { FC } from 'react';
import { PageInfo, Product } from './AddProductItemsModal';
import ShowAttributesButton from '../buttons/ShowAttributesButton';
import { useI18n } from 'mystique/hooks/useI18n';
import { OrderItem } from '../../forms/fields/order/СreateOrder/CreateOrder';

interface AddProductItemsModalListProps {
  loading: boolean;
  products: Product[];
  selectedProducts: OrderItem[];
  pageHistory: string[];
  prevPage: () => void;
  nextPage: () => void;
  pageInfo?: PageInfo;
  addItem: (value: Product) => void;
}

const AddProductItemsModalList: FC<AddProductItemsModalListProps> = ({
  loading,
  products,
  pageHistory,
  prevPage,
  nextPage,
  pageInfo,
  addItem,
  selectedProducts,
}) => {
  const showPagination =
    pageHistory.length > 0 || (products && products.length >= 5);

  const { translate } = useI18n();

  const classes = useStyles();

  return (
    <TableContainer>
      {loading && <Loading />}
      <Table>
        <TableHead>
          <TableRow>
            <TableCell>
              <span>
                {translate(
                  'fc.se.add.product.items.add.modal.table.head.image',
                )}
              </span>
            </TableCell>
            <TableCell>
              <span>
                {translate(
                  'fc.se.add.product.items.add.modal.table.head.product',
                )}
              </span>
            </TableCell>
            <TableCell align={'center'}>
              <span>
                {translate(
                  'fc.se.add.product.items.add.modal.table.head.attributes',
                )}
              </span>
            </TableCell>
            <TableCell align="center">
              <span>
                {translate(
                  'fc.se.add.product.items.add.modal.table.head.quantity',
                )}
              </span>
            </TableCell>
            <TableCell align="center">
              <span>
                {translate('fc.se.add.product.items.add.modal.table.head.add')}
              </span>
            </TableCell>
          </TableRow>
        </TableHead>
        <TableBody>
          {products.length ? (
            products.map((product) => (
              <TableRow key={product.ref}>
                <TableCell>
                  <img
                    src={product.imageUrl}
                    style={{ width: '40px', height: '40px' }}
                  />
                </TableCell>
                <TableCell style={{ color: 'grey' }}>
                  <div style={{ color: 'black' }}>{product.ref}</div>
                  {product.name}
                </TableCell>
                <TableCell align={'center'}>
                  {!!product.name && (
                    <ShowAttributesButton
                      title={
                        '{{i18n "fc.se.add.product.items.add.modal.table.show.attributes.title"}}'
                      }
                      data={product}
                      attributes={[
                        {
                          dataSource: 'attributes',
                          exclude: ['imageUrl'],
                        },
                      ]}
                    />
                  )}
                </TableCell>
                <TableCell align="center">
                  <span>{product.available}</span>
                </TableCell>
                <TableCell align="center">
                  <IconButton
                    color="primary"
                    size="small"
                    disabled={
                      product.available < 1 ||
                      !product.name ||
                      selectedProducts?.some((item) => item.ref === product.ref)
                    }
                    onClick={() => addItem(product)}
                  >
                    <GoPlus />
                  </IconButton>
                </TableCell>
              </TableRow>
            ))
          ) : (
            <TableRow>
              <TableCell colSpan={5} align="center">
                <span>
                  {translate(
                    'fc.se.add.product.items.add.modal.table.no.available',
                  )}
                </span>
              </TableCell>
            </TableRow>
          )}
        </TableBody>
      </Table>
      {showPagination ? (
        <div className={classes.pagination}>
          <span>
            {translate('fc.se.add.product.items.add.modal.showing.rows', {
              start: pageHistory.length * 5 + 1,
              end: pageHistory.length * 5 + products.length,
            })}
          </span>
          <IconButton
            onClick={() => {
              prevPage();
            }}
            disabled={!(pageHistory.length >= 1)}
            style={{
              color:
                pageHistory.length >= 1 ? 'inherit' : 'rgba(0, 0, 0, 0.26)',
            }}
          >
            <ChevronLeft />
          </IconButton>
          <IconButton
            onClick={() => {
              nextPage();
            }}
            disabled={!pageInfo?.hasNextPage}
            style={{
              color: pageInfo?.hasNextPage ? 'inherit' : 'rgba(0, 0, 0, 0.26)',
            }}
          >
            <ChevronRight />
          </IconButton>
        </div>
      ) : null}
    </TableContainer>
  );
};

const useStyles = makeStyles(() =>
  createStyles({
    pagination: {
      display: 'flex',
      justifyContent: 'flex-end',
      alignItems: 'center',
      width: '100%',
      paddingTop: '10px',
    },
  }),
);

export default AddProductItemsModalList;