import { FC } from 'react';
import { TranslatableAttribute } from 'mystique/components/QuantityList';
import { ComponentRegistry } from 'mystique/registry/ComponentRegistry';
import { ListProps } from 'mystique/components/List';
import { useI18n } from 'mystique/hooks/useI18n';
import { OrderItem } from '../../forms/fields/order/СreateOrder/CreateOrder';

const productTableColumns: TranslatableAttribute[] = [
  // product image
  {
    label: 'fc.se.add.product.items.product.list.head.image',
    labelFallback: 'Image',
    type: 'image',
    value: '{{imageUrl}}',
    options: {
      height: 50,
    },
  },
  // product name
  {
    label: 'fc.se.add.product.items.product.list.head.product',
    labelFallback: 'Product',
    type: 'component',
    options: {
      component: 'fc.attribute.column',
      props: {
        value: [
          {
            name: 'name',
          },
          {
            name: 'ref',
            style: {
              color: 'grey',
            },
          },
        ],
      },
    },
  },
  // product price
  {
    label: 'fc.se.add.product.items.product.list.head.price',
    labelFallback: 'Net Price',
    value: '{{currency price prices.0.currency}}',
    align: 'center',
  },
  // Description
  {
    label: 'fc.se.add.product.items.product.list.head.description',
    labelFallback: 'Description',
    type: 'component',
    align: 'center',
    options: {
      component: 'fc.se.button.show.attributes',
      props: {
        title:
          "{{i18n 'fc.se.add.product.items.product.list.attributes.modal.title'}}",
        attributes: [
          {
            dataSource: 'attributes',
            exclude: ['imageUrl', 'imageUrlRef'],
          },
        ],
      },
    },
  },
];

interface AddProductItemsTableProps {
  onDelete: (ref: string) => void;
  onChangeQuantity: (ref: string, name: string, value: any) => void;
  products: OrderItem[];
}

const AddProductItemsTable: FC<AddProductItemsTableProps> = ({
  onChangeQuantity,
  onDelete,
  products,
}) => {
  const { translateOr } = useI18n();

  const FcList = ComponentRegistry.get('fc.list') as FC<ListProps>;

  const modifiedAttributes = [...productTableColumns];

  modifiedAttributes.push({
    label: 'fc.se.add.product.items.product.list.head.quantity',
    labelFallback: 'Quantity',
    align: 'center',
    type: 'component',
    options: {
      component: 'fc.se.quantity.selector',
      props: {
        onChange: (e: any) => onChangeQuantity(e.id, 'quantity', e.value),
        label: '',
        value: { value: '{{quantity}}', id: '{{ref}}' },
        validation: {
          min: 1,
          max: '{{available}}',
        },
        innerText: {
          string: 'i18n:fc.se.add.product.items.product.list.count.ofMax',
          options: {
            color: {
              all: 'error',
              some: 'success',
              none: 'success',
            },
          },
        },
        onBlur: () => null,
      },
    },
  });

  modifiedAttributes.push({
    label: 'fc.se.add.product.items.product.list.head.remove',
    labelFallback: 'Remove',
    align: 'center',
    type: 'component',
    options: {
      component: 'fc.se.remove.button',
      props: {
        onClick: onDelete,
        id: '{{ref}}',
      },
    },
  });

  for (const attr of modifiedAttributes) {
    attr.label = translateOr([attr.label, attr.labelFallback || attr.label]);
  }

  return (
    <FcList
      attributes={modifiedAttributes}
      defaultPageSize={10}
      data={products}
      density={{
        toggle: false,
        initial: 'compressed',
      }}
    />
  );
};

export default AddProductItemsTable;
