import { FC, ReactElement, useState } from 'react';
import {
  Button,
  createStyles,
  makeStyles,
  Typography,
} from '@material-ui/core';
import { useI18n } from 'mystique/hooks/useI18n';

interface WizardProps {
  title?: string;
  steps: stepItem[];
  summary?: WizardSummaryContent;
}

interface stepItem {
  required: boolean;
  content: ReactElement;
  title?: string;
  disabled?: boolean;
  onNext?: () => void;
}

interface WizardSummaryContent {
  content: ReactElement;
  title: string;
}

const Wizard: FC<WizardProps> = ({ title, steps, summary }) => {
  const [activeStep, setActiveStep] = useState<number>(0);
  const classes = useStyles();
  const count = steps.length + (summary ? 1 : 0);
  const { translate } = useI18n();

  const prevStep = () => {
    setActiveStep((prev) => prev - 1);
  };

  const nextStep = () => {
    setActiveStep((prev) => prev + 1);
  };

  return (
    <div>
      <div className={classes.titleBlock}>
        {title && <Typography variant="h4">{title}</Typography>}
        {count && (
          <>
            <Typography variant="h4">
              {summary && activeStep + 1 === count
                ? summary.title
                : `${translate('fc.se.wizard.subtitle', {
                  activeStep: activeStep + 1,
                  count: steps.length,
                })} ${steps[activeStep]?.title}`}
            </Typography>
            <div className={classes.bar}>
              {steps.map((_step, i) => (
                <div
                  key={i}
                  className={`${classes.barItem} ${
                    i === activeStep && classes.barItemActive
                  } ${
                    summary && activeStep + 1 === count && classes.barItemActive
                  }`}
                ></div>
              ))}
            </div>
          </>
        )}
      </div>
      <div className={classes.content}>
        {summary && activeStep + 1 === count
          ? summary.content
          : steps[activeStep]?.content}
      </div>
      <div className={classes.bottom}>
        {activeStep !== 0 && (
          <Button
            variant="outlined"
            onClick={() => {
              prevStep();
            }}
          >
            {translate('fc.se.wizard.button.previous')}
          </Button>
        )}
        {activeStep + 1 !== count && (
          <Button
            onClick={() => {
              nextStep();
              // @ts-ignore
              steps[activeStep].onNext && steps[activeStep].onNext();
            }}
            variant="contained"
            color="primary"
            className={classes.btnNext}
            disabled={
              steps[activeStep]?.required ? steps[activeStep]?.disabled : false
            }
          >
            {steps[activeStep]?.required
              ? translate('fc.se.wizard.button.next')
              : translate('fc.se.wizard.button.skip')}
          </Button>
        )}
      </div>
    </div>
  );
};

const useStyles = makeStyles(() =>
  createStyles({
    titleBlock: {
      boxShadow: '0px 4px 20px rgba(0, 0, 0, 0.11)',
      padding: '20px 0',
      textAlign: 'center',
    },
    bar: {
      paddingTop: '20px',
      display: 'flex',
      flexWrap: 'wrap',
      justifyContent: 'center',
      gap: '4px',
      borderTop: '1px solid #D4D6D8',
      marginTop: '14px',
    },
    barItem: {
      width: '60px',
      height: '6px',
      background: '#D4D6D8',
      borderRadius: '100px',
    },
    barItemActive: {
      background: '#00A9E0',
    },
    content: {
      marginTop: '24px',
    },
    bottom: {
      paddingTop: '24px',
      display: 'flex',
    },
    btnNext: {
      marginLeft: 'auto',
    },
  }),
);

export default Wizard;
