package com.fluentcommerce.connect.custom;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fluentcommerce.connect.core.config.data.RouterMappingSettings;
import com.fluentcommerce.connect.core.configuration.data.AccountReference;
import com.fluentcommerce.connect.core.exception.UnprocessableMessageException;
import com.fluentcommerce.connect.core.listener.data.ConnectSDKMessage;
import com.fluentcommerce.connect.core.listener.data.ExternalMessage;
import com.fluentcommerce.connect.core.listener.external.BaseExternalListener;
import com.fluentcommerce.connect.core.listener.external.ListenerInfo;
import com.fluentcommerce.connect.custom.data.WeatherData;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.Optional;
import java.util.UUID;

@Slf4j
@Component
@ListenerInfo(id = "demo-weather", messageClass = WeatherData.class)
public class DemoQueueListener extends BaseExternalListener<WeatherData> {

    //These values identify your external system and are used to find a Fluent account-retailer during the account map lookup
    //The queue is fixed to a Fluent account (as defined at application-connector.yml) but the retailer is flexible
    private final static String SAMPLE_FIXED_ID = "fluent-ct-dev-2";
    private final static String EXTERNAL_ID = "projectKey";

    @Override
    public @NotNull Optional<ConnectSDKMessage> receiveMessage(@NotNull final ExternalMessage message) throws UnprocessableMessageException {
        final WeatherData weatherData = getMessageContent(message);
        final Optional<AccountReference> accountReference = getFluentAccountReference(Collections.singletonMap(EXTERNAL_ID, SAMPLE_FIXED_ID));
        final Optional<String> messageRoute = getMessageRoute(weatherData);

        if (accountReference.isPresent() && messageRoute.isPresent()) {
            try {
                return Optional.of(ConnectSDKMessage.builder()
                        .id(UUID.randomUUID().toString())
                        .name(messageRoute.get())
                        .accountId(accountReference.get().accountId())
                        .retailerId(accountReference.get().retailerId())
                        .payload(ConnectSDKMessage.toJson(weatherData))
                        .build());
            } catch (final JsonProcessingException e) {
                throw new UnprocessableMessageException(e);
            }
        }else {
            log.warn("Unable to determine account [{}] or route [{}].", accountReference, messageRoute);
        }
        return Optional.empty();
    }

    /**
     * Finds a message route based on the name property as defined by the route mappings at application-connector.yml
     */
    private Optional<String> getMessageRoute(final WeatherData weatherData) {
        return getRouteMappings().stream()
                .filter(route -> weatherData.id().equalsIgnoreCase(route.getProp("id", StringUtils.EMPTY)))
                .findFirst().map(RouterMappingSettings::getRoute);
    }
}
