package com.fluentcommerce.connect.custom.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fluentcommerce.connect.core.exception.UnprocessableMessageException;
import com.fluentcommerce.connect.core.handler.context.HandlerContext;
import com.fluentcommerce.connect.core.utils.ConversionUtils;
import com.fluentcommerce.connect.custom.data.WeatherData;
import com.fluentcommerce.graphql.queries.settings.CreateSettingMutation;
import com.fluentcommerce.graphql.queries.settings.UpdateSettingMutation;
import com.fluentcommerce.graphql.type.CreateSettingInput;
import com.fluentcommerce.graphql.type.UpdateSettingInput;
import com.fluentcommerce.util.core.SettingUtils;
import com.fluentretail.api.exception.FluentApiException;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.Map;

public class UpsertSettingUtils {
    public static final String FLUENT_SETTING_KEY = "fc.connect.weather-update-demo.current-weather";

    private UpsertSettingUtils() {
    }

    public static void upsertSetting(final WeatherData weatherData, final HandlerContext context, final String settingKey) {

        final var settings = SettingUtils.getSettings(context.ofFluentContext(), Collections.singletonMap(settingKey, settingKey));
        if (settings.containsKey(settingKey) && StringUtils.isNotBlank(settings.get(settingKey).getId())) {
            update(settings.get(settingKey).getId(), weatherData, context, settingKey);
        } else {
            create(weatherData, context, settingKey);
        }
    }

    private static void create(final WeatherData weatherData, final HandlerContext context, final String settingKey) {
        final CreateSettingMutation mutation;
        try {
            mutation = CreateSettingMutation.builder()
                    .input(CreateSettingInput.builder()
                            .name(settingKey)
                            .valueType("JSON")
                            .lobValue(convertObject(ConversionUtils.getObjectMapper().writeValueAsString(weatherData)))
                            .context("RETAILER")
                            .contextId(Integer.parseInt(context.getAccountReference().retailerId()))
                            .build())
                    .build();
        } catch (JsonProcessingException e) {
            throw new UnprocessableMessageException(e);
        }
        final CreateSettingMutation.Data result = context.ofFluentContext().executeMutation(mutation, CreateSettingMutation.Data.class);
        if (result == null || result.createSetting() == null || StringUtils.isBlank(result.createSetting().id())) {
            throw new FluentApiException(0, "Fluent mutation failed.");
        }
    }

    private static void update(final String id, final WeatherData weatherData, final HandlerContext context, final String settingKey) {
        final UpdateSettingMutation mutation;
        try {
            mutation = UpdateSettingMutation.builder()
                    .input(UpdateSettingInput.builder()
                            .id(id)
                            .name(settingKey)
                            .valueType("JSON")
                            .lobValue(convertObject(ConversionUtils.getObjectMapper().writeValueAsString(weatherData)))
                            .context("RETAILER")
                            .contextId(Integer.parseInt(context.getAccountReference().retailerId()))
                            .build())
                    .build();
        } catch (JsonProcessingException e) {
            throw new UnprocessableMessageException(e);
        }
        final UpdateSettingMutation.Data result = context.ofFluentContext().executeMutation(mutation, UpdateSettingMutation.Data.class);
        if (result == null || result.updateSetting() == null) {
            throw new FluentApiException(0, "Fluent mutation failed.");
        }
    }

    private static Map convertObject(@NotNull final String value) throws JsonProcessingException {
        return ConversionUtils.getObjectMapper().readValue(value, Map.class);
    }
}
