package com.fluentcommerce.rule.order;

import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.DATE;
import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.STRING;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.ORDER;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_COMPLETE_DATE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_DATE_LIMIT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_PERIOD;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_TYPE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.DEFAULT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.RMA;

import com.fluentcommerce.se.common.graphql.mutations.order.UpdateOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.fulfilment.GetFulfilmentByIdQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.Attribute2;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemEdge;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemNode;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.UpdateOrderInput;
import com.fluentcommerce.se.common.graphql.type.UpdateOrderItemWithOrderInput;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;

@RuleInfo(name = "CalculateReturnDateLimitForOrderItems",
    description = "Calculates Return Date Limit Attribute for all Items in the Order according to Return Period Attribute",
    accepts = {@EventInfo(entityType = ORDER), @EventInfo(entityType = FULFILMENT)})
@Slf4j
public class CalculateReturnDateLimitForOrderItems implements Rule {

    private static final String CLASS_NAME = CalculateReturnDateLimitForOrderItems.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        String orderRef;
        Set<String> orderItemIds;
        if (context.getEntity().getEntityType().equals(FULFILMENT)) {
            GetFulfilmentByIdQuery.Data fulfilment = (GetFulfilmentByIdQuery.Data) context.api()
                .query(GetFulfilmentByIdQuery.builder()
                    .id(context.getEntity().getId())
                    .fulfilmentItemsCount(DEFAULT_PAGE_SIZE)
                    .build());
            if (fulfilment == null || fulfilment.fulfilment() == null || fulfilment.fulfilment().items() == null
                || fulfilment.fulfilment().items().edges().isEmpty()) {
                String message = MessageFormat.format(
                    "{0} - Fulfilment with ref {1} has not been found or it has no items!",
                    logPrefix,
                    context.getEvent().getEntityRef());
                log.error(message);
                throw new RuleExecutionException(message, context.getEvent());
            }
            orderRef = fulfilment.fulfilment().order().ref();
            orderItemIds = fulfilment.fulfilment()
                .items()
                .edges()
                .stream()
                .map(edge -> edge.node().orderItem().id())
                .collect(Collectors.toSet());
        } else {
            orderRef = context.getEntity().getRef();
            orderItemIds = null;
        }

        GetOrderByRefQuery.Data order = (GetOrderByRefQuery.Data) context.api()
            .query(GetOrderByRefQuery.builder()
                .ref(orderRef)
                .includeOrderItems(true)
                .includeAttributes(true)
                .includeCustomer(false)
                .includeFulfilments(false)
                .includeFulfilmentChoice(false)
                .includeArticles(false)
                .includeConsignmentArticles(false)
                .includeFinancialTransactions(false)
                .orderItemCount(DEFAULT_PAGE_SIZE)
                .build());

        if (order == null || order.order() == null || order.order().items() == null
            || order.order().items().itemEdges() == null) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found or it has no items!",
                logPrefix,
                orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        List<UpdateOrderItemWithOrderInput> items = new ArrayList<>();
        final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");

        for (ItemEdge edge : order.order().items().itemEdges()) {
            ItemNode item = edge.itemNode();
            int days = findReturnPeriodAttributeValue(item);
            if ((orderItemIds == null || orderItemIds.contains(item.id())) && isItemReturnable(item) && days > 0) {
                Date date = DateUtils.addDays(new Date(), days);
                date = DateUtils.addMilliseconds(DateUtils.ceiling(date, Calendar.DATE), -1);
                items.add(UpdateOrderItemWithOrderInput.builder()
                    .id(item.id())
                    .attributes(Collections.singletonList(AttributeInput.builder()
                        .name(ATTR_RETURN_DATE_LIMIT)
                        .type(STRING)
                        .value(sdf.format(date))
                        .build()))
                    .build());
            }
        }
        UpdateOrderMutation mutation = UpdateOrderMutation.builder()
            .input(UpdateOrderInput.builder()
                .id(order.order().id())
                .items(items)
                .attributes(Collections.singletonList(AttributeInput.builder()
                    .name(ATTR_COMPLETE_DATE)
                    .type(DATE)
                    .value(new Date())
                    .build()))
                .build())
            .build();
        context.action().mutation(mutation);
    }

    private int findReturnPeriodAttributeValue(ItemNode item) {
        Object value = findAttributeValue(item, ATTR_RETURN_PERIOD);
        if (value != null) {
            try {
                return Integer.parseInt(value.toString());
            } catch (NumberFormatException ignored) {
            }
        }
        return 0;
    }

    private boolean isItemReturnable(ItemNode item) {
        Object value = findAttributeValue(item, ATTR_RETURN_TYPE);
        return RMA.equals(value) || DEFAULT.equals(value);
    }

    private Object findAttributeValue(ItemNode item, String attributeName) {
        if (item.attributes() != null) {
            List<Object> values = item.attributes()
                .stream()
                .filter(it -> it.name().equals(attributeName))
                .map(Attribute2::value)
                .collect(Collectors.toList());
            return values.size() > 0 ? values.get(0) : null;
        }
        return null;
    }
}
