package com.fluentcommerce.rule.order;

import static com.fluentretail.se.plugins.util.Constants.ATTR_SUBSTITUTION_QUANTITY;
import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.INTEGER;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.ORDER;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURNABLE_QTY;

import com.fluentcommerce.se.common.graphql.mutations.order.UpdateOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.Attribute2;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemEdge;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.UpdateOrderInput;
import com.fluentcommerce.se.common.graphql.type.UpdateOrderItemWithOrderInput;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.AttributeUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@RuleInfo(
    name = "CalculateReturnableQuantityForOrderItems",
    description = "Calculates initial returnable quantity value and saves it into " + ATTR_RETURNABLE_QTY
        + " for each order item",
    accepts = {
        @EventInfo(entityType = ORDER)
    }
)
@Slf4j
public class CalculateReturnableQuantityForOrderItems implements Rule {

    private static final String CLASS_NAME = CalculateReturnableQuantityForOrderItems.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        final String orderRef = context.getEntity().getRef();
        if (StringUtils.isEmpty(orderRef)) {
            String message = MessageFormat.format("{0} - Order ref is absent!", logPrefix);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        GetOrderByRefQuery.Data data = (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
            .ref(orderRef)
            .includeAttributes(true)
            .includeOrderItems(true)
            .orderItemCount(DEFAULT_PAGE_SIZE)
            .build());
        if (data.order() == null || data.order().items() == null || data.order().items().itemEdges() == null) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found!",
                logPrefix, orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        List<UpdateOrderItemWithOrderInput> itemsToUpdate = new ArrayList<>();
        for (ItemEdge item : data.order().items().itemEdges()) {
            int returnableQty = item.itemNode().quantity();
            Optional<Attribute2> substitutedQty = AttributeUtils.getAttribute(ATTR_SUBSTITUTION_QUANTITY,
                item.itemNode().attributes());
            if (substitutedQty.isPresent()) {
                returnableQty -= Integer.parseInt(substitutedQty.get().value().toString());
            }
            itemsToUpdate.add(UpdateOrderItemWithOrderInput.builder()
                .id(item.itemNode().id())
                .attributes(Collections.singletonList(AttributeInput.builder()
                    .name(ATTR_RETURNABLE_QTY)
                    .type(INTEGER)
                    .value(returnableQty)
                    .build()))
                .build());
        }

        if (itemsToUpdate.size() > 0) {
            UpdateOrderMutation mutation = UpdateOrderMutation.builder()
                .input(UpdateOrderInput.builder()
                    .id(data.order().id())
                    .items(itemsToUpdate)
                    .build())
                .build();
            log.info(MessageFormat.format("{0} - Performing mutation", logPrefix));
            context.action().mutation(mutation);
        }
    }
}
