package com.fluentcommerce.rule.order;


import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_PERIOD;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_TYPE;

import com.fluentcommerce.se.common.graphql.mutations.order.UpdateOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.fulfilment.GetFulfilmentItemsQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.Attribute1;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemEdge;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemNode;
import com.fluentcommerce.se.common.graphql.queries.product.standardProduct.GetStandardProductQuery;
import com.fluentcommerce.se.common.graphql.queries.product.standardProduct.GetStandardProductQuery.Attribute;
import com.fluentcommerce.se.common.graphql.queries.product.variantProduct.GetVariantProductQuery;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.UpdateOrderInput;
import com.fluentcommerce.se.common.graphql.type.UpdateOrderItemWithOrderInput;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "CopyReturnAttributesToOrderItems",
    description = "Copies Return Type and Period Attributes from Variant/Standard Product to Order Items " +
        "if they exist for the current Fulfilment",
    accepts = {
        @EventInfo(entityType = FULFILMENT)
    }
)
@Slf4j
public class CopyReturnAttributesToOrderItems implements Rule {

    private static final String CLASS_NAME = CopyReturnAttributesToOrderItems.class.getSimpleName();

    private static final String[] ATTRIBUTES = {ATTR_RETURN_TYPE, ATTR_RETURN_PERIOD};

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        final Set<String> orderItemIds = getOrderItemIdsForFulfilment(context, context.getEvent().getEntityId());

        final String orderRef = context.getEvent().getRootEntityRef();
        GetOrderByRefQuery.Data order = (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
            .ref(orderRef)
            .includeOrderItems(true)
            .includeAttributes(true)
            .includeCustomer(false)
            .includeFulfilments(false)
            .includeFulfilmentChoice(false)
            .includeArticles(false)
            .includeConsignmentArticles(false)
            .includeFinancialTransactions(false)
            .orderItemCount(DEFAULT_PAGE_SIZE)
            .build());

        if (order == null || order.order() == null || order.order().items() == null
            || order.order().items().itemEdges() == null) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found or it has no items!",
                logPrefix, orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        List<UpdateOrderItemWithOrderInput> items = new ArrayList<>();

        for (ItemEdge edge : order.order().items().itemEdges()) {
            ItemNode item = edge.itemNode();
            if (!orderItemIds.contains(item.id())) {
                continue;
            }

            List<AttributeInput> attrs = new ArrayList<>();
            for (String attributeName : ATTRIBUTES) {
                AttributeInput attr = findAttributeInVariantProduct(context, item, attributeName);
                if (attr != null) {
                    attrs.add(attr);
                }
            }
            if (attrs.size() > 0) {
                items.add(UpdateOrderItemWithOrderInput.builder().id(item.id()).attributes(attrs).build());
            }
        }

        if (items.size() > 0) {
            UpdateOrderMutation mutation = UpdateOrderMutation.builder()
                .input(UpdateOrderInput.builder()
                    .id(order.order().id())
                    .items(items)
                    .build())
                .build();
            context.action().mutation(mutation);
        }
    }

    private AttributeInput findAttributeInVariantProduct(Context context, ItemNode item, String attributeName) {
        if (item.product() == null || item.product().asVariantProduct() == null
            || item.product().asVariantProduct().attributes() == null) {
            return null;
        }
        for (Attribute1 attr : item.product().asVariantProduct().attributes()) {
            if (attributeName.equals(attr.name())) {
                return AttributeInput.builder()
                    .name(attr.name())
                    .type(attr.type())
                    .value(attr.value())
                    .build();
            }
        }
        return findAttributeInStandardProduct(context, item, attributeName);
    }

    private AttributeInput findAttributeInStandardProduct(Context context, ItemNode item, String attributeName) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        final String variantProductRef = item.product().asVariantProduct().ref();
        final String catalogueRef = item.product().asVariantProduct().catalogue().ref();
        GetVariantProductQuery.Data variantProduct =
            (GetVariantProductQuery.Data) context.api().query(GetVariantProductQuery.builder()
                .ref(variantProductRef)
                .catalogueRef(catalogueRef)
                .build());

        if (variantProduct == null || variantProduct.variantProduct() == null
            || variantProduct.variantProduct().product() == null) {
            log.info(MessageFormat.format("{0} - Variant Product with ref {1} has not been found!",
                logPrefix, variantProduct));
            return null;
        }
        final String standardProductRef = variantProduct.variantProduct().product().ref();
        GetStandardProductQuery.Data standardProduct =
            (GetStandardProductQuery.Data) context.api().query(GetStandardProductQuery.builder()
                .ref(standardProductRef)
                .catalogue(catalogueRef)
                .build());
        if (standardProduct == null || standardProduct.standardProduct() == null) {
            log.info(MessageFormat.format("{0} - Standard Product with ref {1} has not been found!",
                logPrefix, standardProduct));
            return null;
        }
        if (standardProduct.standardProduct().attributes() != null) {
            for (Attribute attr : standardProduct.standardProduct().attributes()) {
                if (attributeName.equals(attr.name())) {
                    return AttributeInput.builder()
                        .name(attr.name())
                        .type(attr.type())
                        .value(attr.value())
                        .build();
                }
            }
        }
        return null;
    }

    private Set<String> getOrderItemIdsForFulfilment(Context context, String fulfilmentId) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        Set<String> ids = new HashSet<>();

        GetFulfilmentItemsQuery.Data data = (GetFulfilmentItemsQuery.Data) context.api()
            .query(GetFulfilmentItemsQuery.builder()
                .id(fulfilmentId)
                .build());

        if (data == null || data.fulfilment() == null || data.fulfilment().items() == null
            || data.fulfilment().items().itemEdges() == null) {
            String message = MessageFormat.format(
                "{0} - Fulfilment with ID {1} has not been found or it has no items!", logPrefix, fulfilmentId);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        data.fulfilment().items().itemEdges().stream()
            .filter(it -> it.itemNode() != null && it.itemNode().orderItem() != null &&
                it.itemNode().orderItem().id() != null)
            .forEach(it -> ids.add(it.itemNode().orderItem().id()));
        return ids;
    }
}

