package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.STRING;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.FULFILMENT_PRICE_ATTRIBUTE;
import static com.fluentretail.se.plugins.util.Constants.FULFILMENT_TAX_PRICE_ATTRIBUTE;
import static com.fluentretail.se.plugins.util.v2.OrderUtils.allItemsReturned;

import com.fluentcommerce.se.common.graphql.mutations.returnorder.UpdateReturnOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.type.AmountTypeInput;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.UpdateReturnOrderInput;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.v2.OrderUtils;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "ApplyFulfilmentCostToReturnOrder",
    description = "Updates the return order with the fulfilment cost included if all eligible items have been returned.",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)

@Slf4j
public class ApplyFulfilmentCostToReturnOrder implements Rule {

    @Override
    public void run(Context context) {
        Event event = context.getEvent();

        // Retrieve the return order details
        GetReturnOrderQuery getReturnOrderQuery = GetReturnOrderQuery.builder()
            .ref(event.getEntityRef())
            .retailer(RetailerId.builder().id(event.getRetailerId()).build())
            .build();
        GetReturnOrderQuery.Data returnOrderData = (GetReturnOrderQuery.Data) context.api().query(getReturnOrderQuery);
        GetReturnOrderQuery.ReturnOrder returnOrder = returnOrderData.returnOrder();

        // Retrieve the full order details (including items with attributes)
        String orderRef = returnOrder.order().ref();
        GetOrderByRefQuery.Data orderData = OrderUtils.getOrderByRefQuery(
            context, orderRef,
            true,
            false,
            true,
            false,
            true,
            false,
            false,
            false
        );
        GetOrderByRefQuery.Order order = orderData.order();

        // Assess if all eligible items have been returned
        if (allItemsReturned(order)) {
            double subTotalAmount = returnOrder.subTotalAmount().amount();
            double totalTaxAmount = returnOrder.totalTax().amount();
            double totalAmount = returnOrder.totalAmount().amount();

            double fulfilmentPrice = 0.0;
            double fulfilmentTaxPrice = 0.0;
            if (order.fulfilmentChoice().fulfilmentPrice() != null) {
                fulfilmentPrice = order.fulfilmentChoice().fulfilmentPrice();
            }
            if (order.fulfilmentChoice().fulfilmentTaxPrice() != null) {
                fulfilmentTaxPrice = order.fulfilmentChoice().fulfilmentTaxPrice();
            }
            log.info(MessageFormat.format("Refund fulfilment cost: {0} with tax {1}",
                fulfilmentPrice,
                fulfilmentTaxPrice));

            // Add the costs to the running totals
            subTotalAmount += fulfilmentPrice;
            totalTaxAmount += fulfilmentTaxPrice;

            totalAmount += fulfilmentPrice;
            totalAmount += fulfilmentTaxPrice;
            log.info(MessageFormat.format("Subtotal :{0} Total Tax: {1} Total Amount: {2}",
                subTotalAmount,
                totalTaxAmount,
                totalAmount));

            // Update the Return Order with the added fulfilment cost and an attribute for fulfilment cost refund
            List<AttributeInput> attributes = new ArrayList<>();
            attributes.add(AttributeInput.builder()
                .name(FULFILMENT_PRICE_ATTRIBUTE)
                .type(STRING)
                .value(fulfilmentPrice)
                .build());
            attributes.add(
                AttributeInput.builder()
                    .name(FULFILMENT_TAX_PRICE_ATTRIBUTE)
                    .type(STRING)
                    .value(fulfilmentTaxPrice)
                    .build());

            UpdateReturnOrderInput updateReturnOrderInput = UpdateReturnOrderInput.builder()
                .retailer(RetailerId.builder().id(event.getRetailerId()).build())
                .ref(returnOrder.ref())
                .subTotalAmount(AmountTypeInput.builder().amount(subTotalAmount).build())
                .totalTax(AmountTypeInput.builder().amount(totalTaxAmount).build())
                .totalAmount(AmountTypeInput.builder().amount(totalAmount).build())
                .attributes(attributes)
                .build();
            UpdateReturnOrderMutation updateReturnOrderMutation = UpdateReturnOrderMutation.builder()
                .input(updateReturnOrderInput)
                .build();
            context.action().mutation(updateReturnOrderMutation);
        } else {
            log.info("No refund of fulfilment cost");
        }
    }
}

