package com.fluentcommerce.rule.order.returnorder;

import com.fluentcommerce.se.common.graphql.mutations.returnorder.UpdateReturnOrderMutation;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.UpdateReturnOrderInput;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.Constants.EntityType;
import com.fluentretail.se.plugins.util.Constants.ReturnOrder;
import com.fluentretail.se.plugins.util.RuleUtils;
import com.fluentretail.se.plugins.util.SettingUtils;
import java.text.MessageFormat;
import java.util.Calendar;
import java.util.UUID;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(name = "AssignRmaToReturnOrder",
    description = "Generate RMA for ReturnOrder and save it to returnAuthorizationKey field",
    accepts = {
        @EventInfo(entityType = EntityType.RETURN_ORDER)
    })
@Slf4j
public class AssignRmaToReturnOrder implements Rule {

    private static final String CLASS_NAME = AssignRmaToReturnOrder.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        String returnOrderRef = context.getEntity().getRef();
        String retailerId = context.getEvent().getRetailerId();

        Integer expiryOnDays = SettingUtils.getSettingValue(context, ReturnOrder.RMA_EXPIRY_ON_DAYS_CONFIG,
            ReturnOrder.DEFAULT_RMA_EXPIRATION_PERIOD_DAYS);

        log.info(MessageFormat.format("{0} - RMA expiry on days: {1}", logPrefix, expiryOnDays));

        String rma = UUID.randomUUID().toString();
        Calendar expirationPeriod = Calendar.getInstance();
        expirationPeriod.add(Calendar.DAY_OF_YEAR, expiryOnDays);
        expirationPeriod.set(Calendar.HOUR_OF_DAY, 23);
        expirationPeriod.set(Calendar.MINUTE, 59);
        expirationPeriod.set(Calendar.SECOND, 59);
        String expirationDate = expirationPeriod.toInstant().toString();

        log.info(
            MessageFormat.format("{0} - Generated RMA: {1}, expiration date: {2}", logPrefix, rma, expirationDate));

        context.action().mutation(UpdateReturnOrderMutation.builder()
            .input(UpdateReturnOrderInput.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .returnAuthorisationKey(rma)
                .returnAuthorisationKeyExpiry(expirationPeriod.getTime())
                .build()
            )
            .build());
    }
}
