package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.CREDIT_MEMO_ITEM;
import static com.fluentretail.se.plugins.util.Constants.EVENT_ATTR_CREDIT_REASON_CODE;
import static com.fluentretail.se.plugins.util.Constants.EVENT_ATTR_ENTITY_REFERENCE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.BILLING_ACCOUNT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.CREDIT_MEMO;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.INVOICE;
import static com.fluentretail.se.plugins.util.Constants.PROP_CREDIT_MEMO_ITEM_TYPE;
import static com.fluentretail.se.plugins.util.Constants.PROP_CREDIT_MEMO_TYPE;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import com.fluentcommerce.se.common.graphql.mutations.creditmemo.CreateCreditMemoMutation;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Attribute;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Node;
import com.fluentcommerce.se.common.graphql.type.AmountTypeInput;
import com.fluentcommerce.se.common.graphql.type.BillingAccountKey;
import com.fluentcommerce.se.common.graphql.type.CreateCreditMemoInput;
import com.fluentcommerce.se.common.graphql.type.CreateCreditMemoItemWithCreditMemoInput;
import com.fluentcommerce.se.common.graphql.type.CurrencyKey;
import com.fluentcommerce.se.common.graphql.type.InvoiceKey;
import com.fluentcommerce.se.common.graphql.type.OrderItemLinkInput;
import com.fluentcommerce.se.common.graphql.type.OrderLinkInput;
import com.fluentcommerce.se.common.graphql.type.ProductCatalogueKey;
import com.fluentcommerce.se.common.graphql.type.ProductKey;
import com.fluentcommerce.se.common.graphql.type.QuantityTypeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.ReturnOrderItemKey;
import com.fluentcommerce.se.common.graphql.type.ReturnOrderKey;
import com.fluentcommerce.se.common.graphql.type.SettingValueTypeInput;
import com.fluentcommerce.se.common.graphql.type.TaxTypeInput;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.model.oms.EntityReference;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "CreateCreditMemoFromReturnOrder",
    description =
        "Creates a CreditMemo for the BillingAccount using the supplied ReturnOrder details. This rule has the following two props: {"
            + PROP_CREDIT_MEMO_TYPE + "} and {" + PROP_CREDIT_MEMO_ITEM_TYPE
            + "}. If they are not provided it will take the default values: '"
            + CREDIT_MEMO + "' and '" + CREDIT_MEMO_ITEM + "'.",
    accepts = {
        @EventInfo(
            entityType = BILLING_ACCOUNT
        )
    }
)
@ParamString(name = PROP_CREDIT_MEMO_TYPE,
    description = "Credit memo type",
    defaultValue = CREDIT_MEMO)
@ParamString(name = PROP_CREDIT_MEMO_ITEM_TYPE,
    description = "Credit memo item type",
    defaultValue = CREDIT_MEMO_ITEM)
@Slf4j
public class CreateCreditMemoFromReturnOrder implements Rule {

    private static final String CLASS_NAME = CreateCreditMemoFromReturnOrder.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        final String accountId = context.getEvent().getAccountId();
        final String retailerId = context.getEvent().getRetailerId();
        final String billingAccountRef = context.getEvent().getEntityRef();

        try {
            if (isEmpty(retailerId)) {
                throw new IllegalArgumentException("RetailerId is missing in the incoming event info");
            }
            if (isEmpty(billingAccountRef)) {
                throw new IllegalArgumentException("BillingAccountRef is missing in the incoming event info");
            }

            final String creditMemoType = context.getProp(PROP_CREDIT_MEMO_TYPE);
            final String creditMemoItemType = context.getProp(PROP_CREDIT_MEMO_ITEM_TYPE);

            final String returnOrderRef = getReturnOrderRefFromAttributes(context.getEvent());
            GetReturnOrderQuery.Data data = (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .includeAttributes(true)
                .includeReturnOrderItems(true)
                .includePickupLocation(true)
                .build());

            if (data == null || data.returnOrder() == null) {
                throw new IllegalArgumentException(String.format("ReturnOrder was not found with ref: %s",
                    returnOrderRef));
            }

            Set<GetReturnOrderQuery.Node> nodeSet = getAllReturnOrderItems(data, retailerId);
            if (nodeSet == null || nodeSet.isEmpty()) {
                throw new IllegalArgumentException(String.format("ReturnOrder with ref: '%s' has no ReturnOrderItems",
                    returnOrderRef));
            }

            GetReturnOrderQuery.ReturnOrder returnOrder = data.returnOrder();
            validateMandatoryFields(returnOrder);

            final String creditMemoRef = UUID.randomUUID().toString();

            CreateCreditMemoInput.Builder input = CreateCreditMemoInput.builder()
                .ref(creditMemoRef)
                .type(creditMemoType)
                .billingAccount(BillingAccountKey.builder()
                    .ref(billingAccountRef)
                    .build())
                .items(getCreditMemoItemList(context.getEvent(),
                    nodeSet,
                    retailerId,
                    returnOrderRef,
                    creditMemoItemType))
                .returnOrder(ReturnOrderKey.builder()
                    .ref(returnOrderRef)
                    .retailer(RetailerId.builder()
                        .id(retailerId)
                        .build())
                    .build())
                .issueDate(new Date())
                .currency(CurrencyKey.builder()
                    .alphabeticCode(returnOrder.currency().alphabeticCode())
                    .build())
                .defaultTaxType(TaxTypeInput.builder()
                    .country(returnOrder.defaultTaxType().country())
                    .group(returnOrder.defaultTaxType().group())
                    .tariff(returnOrder.defaultTaxType().tariff())
                    .build())
                .subTotalAmount(AmountTypeInput.builder()
                    .amount(returnOrder.subTotalAmount().amount())
                    .scale(returnOrder.subTotalAmount().scale())
                    .unscaledValue(returnOrder.subTotalAmount().unscaledValue())
                    .build())
                .totalTax(AmountTypeInput.builder()
                    .amount(returnOrder.totalTax().amount())
                    .scale(returnOrder.totalTax().scale())
                    .unscaledValue(returnOrder.totalTax().unscaledValue())
                    .build())
                .totalAmount(AmountTypeInput.builder()
                    .amount(returnOrder.totalAmount().amount())
                    .scale(returnOrder.totalAmount().scale())
                    .unscaledValue(returnOrder.totalAmount().unscaledValue())
                    .build())
                .totalBalance(AmountTypeInput.builder()
                    .amount(returnOrder.totalAmount().amount())
                    .scale(returnOrder.totalAmount().scale())
                    .unscaledValue(returnOrder.totalAmount().unscaledValue())
                    .build());

            if (returnOrder.order() != null && !isEmpty(returnOrder.order().ref())) {
                input = input.order(OrderLinkInput.builder()
                    .retailer(RetailerId.builder()
                        .id(retailerId)
                        .build())
                    .ref(returnOrder.order().ref())
                    .build());
            }

            final String returnOrderInvoiceRef = getReturnOrderInvoiceRef(returnOrder);
            if (!isEmpty(returnOrderInvoiceRef)) {
                input.invoice(InvoiceKey.builder().ref(returnOrderInvoiceRef).build());
            }

            CreateCreditMemoMutation mutation = CreateCreditMemoMutation.builder().input(input.build()).build();
            context.action().mutation(mutation);

        } catch (Exception ex) {
            String message = MessageFormat.format("{0} - Exception message: {1}", logPrefix,
                ex.getMessage() == null ? ex.toString() : ex.getMessage());
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }
    }

    private String getReturnOrderRefFromAttributes(Event event) {
        EntityReference returnOrder = event.getAttribute(EVENT_ATTR_ENTITY_REFERENCE, EntityReference.class);

        if (returnOrder == null) {
            throw new IllegalArgumentException("The entityReference is null or doesn't exist");
        }
        if (isEmpty(returnOrder.getRef())) {
            throw new IllegalArgumentException("ReturnOrderRef is missing");
        }
        if (isEmpty(returnOrder.getType()) || !returnOrder.getType().equalsIgnoreCase(RETURN_ORDER)) {
            throw new IllegalArgumentException("The entityReferenced just support type = " + RETURN_ORDER);
        }

        return returnOrder.getRef();
    }

    private Set<GetReturnOrderQuery.Node> getAllReturnOrderItems(GetReturnOrderQuery.Data data, String retailerId) {
        final Set<GetReturnOrderQuery.Node> result = new HashSet<>();
        if (data == null || data.returnOrder() == null || data.returnOrder().returnOrderItems() == null
            || data.returnOrder().returnOrderItems().edges().isEmpty()) {
            return result;
        }
        data.returnOrder().returnOrderItems().edges().stream().map(GetReturnOrderQuery.Edge::node).forEach(result::add);
        return result;
    }

    private SettingValueTypeInput getCreditReasonCodeFromAttributes(Event event) {
        Map<String, Object> valueLabelMap = event.getAttribute(EVENT_ATTR_CREDIT_REASON_CODE, Map.class);
        if (valueLabelMap == null || valueLabelMap.isEmpty()) {
            return null;
        }

        if (valueLabelMap.containsKey("value") && !isEmpty((String) valueLabelMap.get("value"))
            && valueLabelMap.containsKey("label") && !isEmpty((String) valueLabelMap.get("label"))) {
            return SettingValueTypeInput.builder()
                .label((String) valueLabelMap.get("label"))
                .value((String) valueLabelMap.get("value"))
                .build();
        } else {
            return null;
        }
    }

    private List<CreateCreditMemoItemWithCreditMemoInput> getCreditMemoItemList
        (Event event, Set<Node> returnOrderItems, String retailerId, String returnOrderRef, String itemType) {

        List<CreateCreditMemoItemWithCreditMemoInput> itemsList = new LinkedList<>();

        returnOrderItems.forEach(item -> {

            validateItemMandatoryFields(item);

            String ref = UUID.randomUUID().toString();

            CreateCreditMemoItemWithCreditMemoInput.Builder basicItem =
                CreateCreditMemoItemWithCreditMemoInput.builder()
                    .ref(ref)
                    .type(itemType)
                    .returnOrderItem(ReturnOrderItemKey.builder()
                        .ref(item.ref())
                        .returnOrder(ReturnOrderKey.builder()
                            .ref(returnOrderRef)
                            .retailer(RetailerId.builder()
                                .id(retailerId)
                                .build())
                            .build())
                        .build())
                    .unitQuantity(QuantityTypeInput.builder()
                        .quantity(item.unitQuantity().quantity())
                        .unit(item.unitQuantity().unit())
                        .build())
                    .unitAmount(AmountTypeInput.builder()
                        .amount(item.unitAmount().amount())
                        .scale(item.unitAmount().scale())
                        .unscaledValue(item.unitAmount().unscaledValue())
                        .build())
                    .amount(AmountTypeInput.builder()
                        .amount(item.itemAmount().amount())
                        .scale(item.itemAmount().scale())
                        .unscaledValue(item.itemAmount().unscaledValue())
                        .build())
                    .taxAmount(AmountTypeInput.builder()
                        .amount(item.itemTaxAmount().amount())
                        .scale(item.itemTaxAmount().scale())
                        .unscaledValue(item.itemTaxAmount().unscaledValue())
                        .build());

            if (item.orderItem() != null && !isEmpty(item.orderItem().ref())) {
                basicItem = basicItem.orderItem(OrderItemLinkInput.builder()
                    .ref(item.orderItem().ref())
                    .order(OrderLinkInput.builder()
                        .ref(item.orderItem().order().ref())
                        .retailer(RetailerId.builder()
                            .id(retailerId)
                            .build())
                        .build())
                    .build());
            }

            if (item.product() != null && !isEmpty(item.product().ref())) {
                basicItem = basicItem.product(ProductKey.builder()
                    .ref(item.product().ref())
                    .catalogue(ProductCatalogueKey.builder()
                        .ref(item.product().catalogue().ref())
                        .build())
                    .build());
            }

            if (getCreditReasonCodeFromAttributes(event) != null) {
                basicItem = basicItem.creditReasonCode(getCreditReasonCodeFromAttributes(event));
            }

            if (item.unitTaxType() != null && !isEmpty(item.unitTaxType().tariff())) {
                basicItem = basicItem.unitTaxType(TaxTypeInput.builder()
                    .tariff(item.unitTaxType().tariff())
                    .group(item.unitTaxType().group())
                    .country(item.unitTaxType().country())
                    .build());
            }

            itemsList.add(basicItem.build());
        });

        return itemsList;
    }

    private String getReturnOrderInvoiceRef(GetReturnOrderQuery.ReturnOrder returnOrder) {
        Optional<Attribute> attr = returnOrder.attributes().stream()
            .filter(a -> a.name().equalsIgnoreCase(INVOICE)).findFirst();
        return attr.map(attribute -> attribute.value().toString()).orElse(null);
    }

    private void validateMandatoryFields(GetReturnOrderQuery.ReturnOrder returnOrder) {
        if (returnOrder == null) {
            throw new IllegalArgumentException("Return Order has not been found!");
        }

        if (returnOrder.currency() == null || isEmpty(returnOrder.currency().alphabeticCode())) {
            throw new IllegalArgumentException("Currency is missing");
        }

        if (returnOrder.defaultTaxType() == null || isEmpty(returnOrder.defaultTaxType().tariff())) {
            throw new IllegalArgumentException("defaultTaxType is missing");
        }

        if (returnOrder.subTotalAmount() == null || returnOrder.subTotalAmount().amount() == null) {
            throw new IllegalArgumentException("subTotalAmount is missing");
        }

        if (returnOrder.totalTax() == null || returnOrder.totalTax().amount() == null) {
            throw new IllegalArgumentException("totalTax is missing");
        }

        if (returnOrder.totalAmount() == null || returnOrder.totalAmount().amount() == null) {
            throw new IllegalArgumentException("totalAmount is missing");
        }
    }

    private void validateItemMandatoryFields(GetReturnOrderQuery.Node item) {
        if (item.unitQuantity() == null || item.unitQuantity().quantity() == null) {
            throw new IllegalArgumentException("unitQuantity is missing");
        }

        if (item.unitAmount() == null || item.unitAmount().amount() == null) {
            throw new IllegalArgumentException("unitAmount is missing");
        }

        if (item.itemAmount() == null || item.itemAmount().amount() == null) {
            throw new IllegalArgumentException("itemAmount is missing");
        }

        if (item.itemTaxAmount() == null || item.itemTaxAmount().amount() == null) {
            throw new IllegalArgumentException("itemTaxAmount is missing");
        }
    }
}
