package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.INVOICE;
import static com.fluentretail.se.plugins.util.Constants.INVOICE_ITEM_TYPE;
import static com.fluentretail.se.plugins.util.Constants.INVOICE_TYPE_RETURN;
import static com.fluentretail.se.plugins.util.Constants.PROP_STATUS;

import com.fluentcommerce.se.common.graphql.mutations.invoice.CreateInvoiceMutation;
import com.fluentcommerce.se.common.graphql.mutations.returnorder.UpdateReturnOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.billingAccount.GetBillingAccountsByCustomerRefQuery;
import com.fluentcommerce.se.common.graphql.queries.returnfulfilment.GetReturnFulfilmentQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderWithFulfilmentsQuery;
import com.fluentcommerce.se.common.graphql.type.AmountTypeInput;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.BillingAccountKey;
import com.fluentcommerce.se.common.graphql.type.CreateInvoiceInput;
import com.fluentcommerce.se.common.graphql.type.CreateInvoiceItemWithInvoiceInput;
import com.fluentcommerce.se.common.graphql.type.CurrencyKey;
import com.fluentcommerce.se.common.graphql.type.OrderLinkInput;
import com.fluentcommerce.se.common.graphql.type.ProductCatalogueKey;
import com.fluentcommerce.se.common.graphql.type.ProductKey;
import com.fluentcommerce.se.common.graphql.type.QuantityTypeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.TaxTypeInput;
import com.fluentcommerce.se.common.graphql.type.UpdateReturnOrderInput;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.Constants.Attributes.Types;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "CreateInvoiceForReturnOrder",
    description = "Creates Invoice for the Return Order taking into account only fulfilments with statuses: {" +
        PROP_STATUS + "}",
    accepts = {
        @EventInfo(
            entityType = RETURN_ORDER
        )
    }
)
@ParamString(name = PROP_STATUS,
    description = "The list of return accepted fulfilment statuses")
@Slf4j
public class CreateInvoiceForReturnOrder implements Rule {

    private static final String CLASS_NAME = CreateInvoiceForReturnOrder.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        final String retailerId = context.getEvent().getRetailerId();
        final String returnOrderRef = context.getEvent().getEntityRef();
        final Set<String> statuses = new TreeSet<>(context.getPropList(PROP_STATUS, String.class));

        GetReturnOrderWithFulfilmentsQuery.Data data =
            (GetReturnOrderWithFulfilmentsQuery.Data) context.api().query(GetReturnOrderWithFulfilmentsQuery.builder()
                .retailer(RetailerId.builder().id(retailerId).build())
                .ref(returnOrderRef)
                .includePickupLocation(false)
                .includeAttributes(false)
                .includeReturnOrderFulfilments(true)
                .includeReturnOrderItems(true)
                .build());

        if (data == null || data.returnOrder() == null || data.returnOrder().returnOrderItems() == null) {
            String message =
                MessageFormat.format("{0} - Return Order with ref {1} has not been found or it has no items!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder = data.returnOrder();

        final List<GetReturnFulfilmentQuery.ReturnFulfilment> fulfilments = new ArrayList<>();
        returnOrder.returnOrderFulfilments().edges().forEach(rf -> {
            if (statuses.isEmpty() || statuses.contains(rf.node().status())) {
                GetReturnFulfilmentQuery.Data returnFulfilment =
                    (GetReturnFulfilmentQuery.Data) context.api().query(GetReturnFulfilmentQuery.builder()
                        .ref(rf.node().ref())
                        .returnOrderRef(returnOrderRef)
                        .retailer(RetailerId.builder().id(retailerId).build())
                        .includeAttributes(false)
                        .includePickupLocation(false)
                        .includeReturnOrderItems(true)
                        .build());
                if (returnFulfilment != null && returnFulfilment.returnFulfilment() != null) {
                    fulfilments.add(returnFulfilment.returnFulfilment());
                }
            }
        });

        if (fulfilments.isEmpty()) {
            String message = MessageFormat.format("{0} - There are no acceptable fulfilments in the Return Order!",
                logPrefix);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        GetBillingAccountsByCustomerRefQuery.Data billingAccountData =
            (GetBillingAccountsByCustomerRefQuery.Data) context.api()
                .query(GetBillingAccountsByCustomerRefQuery.builder()
                    .customerRef(returnOrder.customer().ref())
                    .build());
        if (billingAccountData == null || billingAccountData.billingAccounts() == null ||
            billingAccountData.billingAccounts().edges() == null || billingAccountData.billingAccounts()
            .edges()
            .isEmpty()) {
            String message = MessageFormat.format("{0} - There are no billing accounts for the customer with Ref {1}",
                logPrefix, returnOrder.customer().ref());
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }
        final String billingAccountRef = billingAccountData.billingAccounts().edges().get(0).node().ref();

        CreateInvoiceMutation createInvoice =
            buildInvoiceMutation(retailerId, returnOrder, fulfilments, billingAccountRef);
        log.info(MessageFormat.format("{0} - Create Invoice: {1}", logPrefix, createInvoice));
        context.action().mutation(createInvoice);

        UpdateReturnOrderMutation updateReturnOrder = UpdateReturnOrderMutation.builder()
            .input(UpdateReturnOrderInput.builder()
                .ref(returnOrder.ref())
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .attributes(Collections.singletonList(AttributeInput.builder()
                    .name(INVOICE)
                    .type(Types.STRING)
                    .value(createInvoice.variables().input().ref())
                    .build()))
                .build())
            .build();
        context.action().mutation(updateReturnOrder);
    }

    private CreateInvoiceMutation buildInvoiceMutation(String retailerId,
        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder,
        List<GetReturnFulfilmentQuery.ReturnFulfilment> fulfilments, String billingAccountRef) {

        CreateInvoiceMutation mutation = CreateInvoiceMutation.builder()
            .input(CreateInvoiceInput.builder()
                .ref(UUID.randomUUID().toString())
                .type(INVOICE_TYPE_RETURN)
                .billingAccount(BillingAccountKey.builder()
                    .ref(billingAccountRef)
                    .build())
                .order(OrderLinkInput.builder()
                    .ref(returnOrder.ref())
                    .retailer(RetailerId.builder().id(retailerId).build())
                    .build())
                .items(buildInvoiceItems(returnOrder, fulfilments))
                .issueDate(new Date())
                .currency(CurrencyKey.builder().alphabeticCode(returnOrder.currency().alphabeticCode()).build())
                .defaultTaxType(TaxTypeInput.builder()
                    .country(returnOrder.defaultTaxType().country())
                    .group(returnOrder.defaultTaxType().group())
                    .tariff(returnOrder.defaultTaxType().tariff())
                    .build())
                .subTotalAmount(AmountTypeInput.builder()
                    .amount(BigDecimal.valueOf(returnOrder.subTotalAmount().amount())
                        .setScale(2, RoundingMode.HALF_UP)
                        .doubleValue())
                    .build())
                .totalTax(AmountTypeInput.builder()
                    .amount(BigDecimal.valueOf(returnOrder.totalTax().amount())
                        .setScale(2, RoundingMode.HALF_UP)
                        .doubleValue())
                    .build())
                .totalAmount(AmountTypeInput.builder()
                    .amount(BigDecimal.valueOf(returnOrder.totalAmount().amount())
                        .setScale(2, RoundingMode.HALF_UP)
                        .doubleValue())
                    .build())
                .totalBalance(AmountTypeInput.builder()
                    .amount(BigDecimal.ZERO.doubleValue())
                    .build())
                .build())
            .build();
        return mutation;
    }

    private List<CreateInvoiceItemWithInvoiceInput> buildInvoiceItems(
        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder,
        List<GetReturnFulfilmentQuery.ReturnFulfilment> fulfilments) {
        final List<CreateInvoiceItemWithInvoiceInput> items = new ArrayList<>();
        returnOrder.returnOrderItems().edges().forEach(it -> {
            Integer quantity = getItemUnitQuantity(it.node(), fulfilments);
            CreateInvoiceItemWithInvoiceInput item = CreateInvoiceItemWithInvoiceInput.builder()
                .ref(UUID.randomUUID().toString())
                .type(INVOICE_ITEM_TYPE)
                .product(ProductKey.builder()
                    .ref(it.node().product().ref())
                    .catalogue(ProductCatalogueKey.builder().ref(it.node().product().catalogue().ref()).build())
                    .build())
                .unitQuantity(QuantityTypeInput.builder()
                    .quantity(quantity)
                    .unit(it.node().unitQuantity().unit())
                    .build())
                .unitTaxType(TaxTypeInput.builder()
                    .country(it.node().unitTaxType().country())
                    .group(it.node().unitTaxType().group())
                    .tariff(it.node().unitTaxType().tariff())
                    .build())
                .unitAmount(AmountTypeInput.builder()
                    .amount(BigDecimal.valueOf(it.node().unitAmount().amount())
                        .setScale(2, RoundingMode.HALF_UP)
                        .doubleValue())
                    .build())
                .amount(AmountTypeInput.builder()
                    .amount(BigDecimal.valueOf(
                            quantity * (it.node().unitAmount().amount() + it.node().itemTaxAmount().amount()))
                        .setScale(2, RoundingMode.HALF_UP)
                        .doubleValue())
                    .build())
                .taxAmount(AmountTypeInput.builder()
                    .amount(BigDecimal.valueOf(it.node().itemTaxAmount().amount())
                        .setScale(2, RoundingMode.HALF_UP)
                        .doubleValue())
                    .build())
                .build();
            items.add(item);
        });
        return items;
    }


    private Integer getItemUnitQuantity(GetReturnOrderWithFulfilmentsQuery.Node item,
        List<GetReturnFulfilmentQuery.ReturnFulfilment> returnFulfilments) {
        int quantity = returnFulfilments.stream()
            .flatMap(rf -> rf.returnFulfilmentItems().edges().stream()
                .filter(fit -> fit.node().returnOrderItem().ref().equalsIgnoreCase(item.ref())))
            .mapToInt(v -> v.node().unitQuantity().quantity()).sum();
        return Math.min(item.unitQuantity().quantity(), quantity);
    }
}
