package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.STRING;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_TYPE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.DEFAULT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.NONE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.RMA;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import com.fluentcommerce.se.common.graphql.mutations.returnfulfilment.CreateReturnFulfilmentMutation;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemEdge;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Attribute;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Attribute1;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Edge;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Node;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.CreateReturnFulfilmentInput;
import com.fluentcommerce.se.common.graphql.type.CreateReturnFulfilmentItemWithReturnFulfilmentInput;
import com.fluentcommerce.se.common.graphql.type.LocationLinkInput;
import com.fluentcommerce.se.common.graphql.type.ProductCatalogueKey;
import com.fluentcommerce.se.common.graphql.type.ProductKey;
import com.fluentcommerce.se.common.graphql.type.QuantityTypeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.ReturnOrderItemKey;
import com.fluentcommerce.se.common.graphql.type.ReturnOrderKey;
import com.fluentcommerce.se.common.graphql.type.StreetAddressInput;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@RuleInfo(
    name = "CreateReturnFulfilments",
    description = "Creates Return Fulfilments for the Current Order",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@Slf4j
public class CreateReturnFulfilments implements Rule {

    private static final String CLASS_NAME = CreateReturnFulfilments.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        final String returnOrderRef = context.getEntity().getRef();
        GetReturnOrderQuery.Data returnOrder =
            (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .includeAttributes(true)
                .includeReturnOrderItems(true)
                .includePickupLocation(true)
                .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null
            || returnOrder.returnOrder().returnOrderItems() == null) {
            String message =
                MessageFormat.format("{0} - Return Order with ref {1} has not been found or it has no items!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        List<Node> defaultItems = new ArrayList<>();
        List<Node> rmaItems = new ArrayList<>();
        final String orderRef = returnOrder.returnOrder().order().ref();
        Map<String, String> returnTypesByOrderItemRef = getReturnTypeByOrderItems(context, orderRef);

        for (Edge edge : returnOrder.returnOrder().returnOrderItems().edges()) {
            Node node = edge.node();
            if (RMA.equals(returnTypesByOrderItemRef.get(node.orderItem().ref()))) {
                rmaItems.add(node);
            } else if (DEFAULT.equals(returnTypesByOrderItemRef.get(node.orderItem().ref()))) {
                defaultItems.add(node);
            }
        }

        if (defaultItems.isEmpty() && rmaItems.isEmpty()) {
            String message =
                MessageFormat.format("{0} - There are no items for both return types!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        if (!defaultItems.isEmpty()) {
            createFulfilment(context, DEFAULT, defaultItems, returnOrder);
            log.info(MessageFormat.format("{0} - Fulfilment for DEFAULT Return has been created!", logPrefix));
        }

        if (!rmaItems.isEmpty()) {
            createFulfilment(context, RMA, rmaItems, returnOrder);
            log.info(MessageFormat.format("{0} - Fulfilment for RMA Return has been created!", logPrefix));
        }
    }

    private Map<String, String> getReturnTypeByOrderItems(Context context, String orderRef) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        GetOrderByRefQuery.Data order = (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
            .ref(orderRef)
            .includeOrderItems(true)
            .includeAttributes(true)
            .includeCustomer(false)
            .includeFulfilments(false)
            .includeFulfilmentChoice(false)
            .includeArticles(false)
            .includeConsignmentArticles(false)
            .includeFinancialTransactions(false)
            .orderItemCount(DEFAULT_PAGE_SIZE)
            .build());

        if (order == null || order.order() == null || order.order().items() == null) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found or it has no items!",
                logPrefix, orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        final Map<String, String> returnTypesByOrderItemRef = new HashMap<>();
        for (ItemEdge edge : order.order().items().itemEdges()) {
            List<String> attrs = (edge.itemNode().attributes() == null) ? null : edge.itemNode().attributes().stream()
                .filter(at -> ATTR_RETURN_TYPE.equals(at.name()))
                .map(at -> at.value().toString())
                .collect(Collectors.toList());

            String returnType = attrs == null || attrs.size() == 0 || StringUtils.isEmpty(attrs.get(0).trim()) ? DEFAULT
                : attrs.get(0).trim().toUpperCase();

            switch (returnType) {
                case DEFAULT:
                case RMA:
                    returnTypesByOrderItemRef.put(edge.itemNode().ref(), returnType);
                    break;
                default:
                    returnTypesByOrderItemRef.put(edge.itemNode().ref(), NONE);
            }
        }

        return returnTypesByOrderItemRef;
    }

    private void createFulfilment(Context context, String returnType, List<Node> items,
        GetReturnOrderQuery.Data returnOrder) throws IllegalArgumentException {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());

        String ref = UUID.randomUUID().toString();
        List<AttributeInput> attributes = convertOrderAttributes(returnOrder.returnOrder().attributes());
        attributes.add(AttributeInput.builder()
            .name(ATTR_RETURN_TYPE)
            .type(STRING)
            .value(returnType)
            .build());
        List<CreateReturnFulfilmentItemWithReturnFulfilmentInput> fulfilmentItems = createReturnFulfilmentItems(items,
            returnOrder.returnOrder().ref(), context.getEvent().getRetailerId());
        StreetAddressInput pickupAddress = convertAddress(returnOrder.returnOrder().pickupAddress());
        String lodgedLocationRef = returnOrder.returnOrder().lodgedLocation() != null
            ? returnOrder.returnOrder().lodgedLocation().ref() : null;
        String destinationLocationRef = (returnOrder.returnOrder().destinationLocation() == null ||
            isEmpty(returnOrder.returnOrder().destinationLocation().ref())) ? lodgedLocationRef
            : returnOrder.returnOrder().destinationLocation().ref();

        CreateReturnFulfilmentMutation mutation = CreateReturnFulfilmentMutation.builder()
            .input(CreateReturnFulfilmentInput.builder()
                .ref(ref)
                .returnOrder(ReturnOrderKey.builder()
                    .ref(returnOrder.returnOrder().ref())
                    .retailer(RetailerId.builder()
                        .id(context.getEvent().getRetailerId())
                        .build())
                    .build())
                .type(returnOrder.returnOrder().type())
                .attributes(attributes)
                .pickupAddress(pickupAddress)
                .lodgedLocation(isEmpty(lodgedLocationRef) ? null : LocationLinkInput.builder()
                    .ref(lodgedLocationRef)
                    .build())
                .destinationLocation(LocationLinkInput.builder()
                    .ref(destinationLocationRef)
                    .build())
                .returnFulfilmentItems(fulfilmentItems)
                .build())
            .build();

        context.action().mutation(mutation);
        log.info(MessageFormat.format("{0} - Fulfilment has been created: {1}", logPrefix, mutation));
    }

    private List<CreateReturnFulfilmentItemWithReturnFulfilmentInput> createReturnFulfilmentItems(List<Node> items,
        String returnOrderRef, String retailerId) throws IllegalArgumentException {
        if (items == null || items.isEmpty()) {
            return null;
        }

        List<CreateReturnFulfilmentItemWithReturnFulfilmentInput> result = new ArrayList<>();

        items.forEach(item -> {
            String ref = UUID.randomUUID().toString();

            String itemRef;
            if (item != null && !isEmpty(item.ref())) {
                itemRef = item.ref();
            } else {
                throw new IllegalArgumentException(
                    String.format("ReturnOrderItemRef is missing for ReturnOrder with ref: %s", returnOrderRef));
            }

            int unitQuantity;
            if (item.unitQuantity() != null && item.unitQuantity().quantity() != null) {
                unitQuantity = item.unitQuantity().quantity();
            } else {
                throw new IllegalArgumentException(
                    String.format("UnitQuantity.quantity is missing for ReturnOrder with ref: %s", returnOrderRef));
            }

            String productRef;
            if (item.product() != null && !isEmpty(item.product().ref())) {
                productRef = item.product().ref();
            } else {
                throw new IllegalArgumentException(
                    String.format("ProductRef is missing for ReturnOrder with ref: %s", returnOrderRef));
            }

            String catalogueRef;
            if (item.product().catalogue() != null && !isEmpty(item.product().catalogue().ref())) {
                catalogueRef = item.product().catalogue().ref();
            } else {
                throw new IllegalArgumentException(
                    String.format("CatalogueRef is missing for ReturnOrder with ref: %s", returnOrderRef));
            }

            String unitQuantityUnit = (item.unitQuantity() != null) ? item.unitQuantity().unit() : "";

            result.add(
                CreateReturnFulfilmentItemWithReturnFulfilmentInput.builder()
                    .ref(ref)
                    .returnOrderItem(ReturnOrderItemKey.builder()
                        .ref(itemRef)
                        .returnOrder(ReturnOrderKey.builder()
                            .ref(returnOrderRef)
                            .retailer(RetailerId.builder()
                                .id(retailerId)
                                .build())
                            .build())
                        .build())
                    .unitQuantity(QuantityTypeInput.builder()
                        .quantity(unitQuantity)
                        .unit(unitQuantityUnit)
                        .build())
                    .attributes(convertOrderItemAttributes(item.attributes()))
                    .product(ProductKey.builder()
                        .ref(productRef)
                        .catalogue(ProductCatalogueKey.builder()
                            .ref(catalogueRef)
                            .build())
                        .build())
                    .build()
            );
        });
        return result;
    }

    private List<AttributeInput> convertOrderAttributes(List<Attribute> attrs) {
        List<AttributeInput> attributes = new ArrayList<>();
        if (attrs != null && attrs.size() > 0) {
            for (Attribute attr : attrs) {
                attributes.add(AttributeInput.builder()
                    .name(attr.name())
                    .type(attr.type())
                    .value(attr.value())
                    .build());
            }
        }
        return attributes;
    }

    private List<AttributeInput> convertOrderItemAttributes(List<Attribute1> attrs) {
        List<AttributeInput> attributes = new ArrayList<>();
        if (attrs != null && attrs.size() > 0) {
            for (Attribute1 attr : attrs) {
                attributes.add(AttributeInput.builder()
                    .name(attr.name())
                    .type(attr.type())
                    .value(attr.value())
                    .build());
            }
        }
        return attributes;
    }

    private StreetAddressInput convertAddress(GetReturnOrderQuery.PickupAddress pickupAddress) {
        if (pickupAddress != null) {
            return StreetAddressInput.builder()
                .companyName(pickupAddress.companyName())
                .name(pickupAddress.name())
                .street(pickupAddress.street())
                .city(pickupAddress.city())
                .state(pickupAddress.state())
                .postcode(pickupAddress.postcode())
                .region(pickupAddress.region())
                .country(pickupAddress.country())
                .latitude(pickupAddress.latitude())
                .longitude(pickupAddress.longitude())
                .timeZone(pickupAddress.timeZone())
                .build();
        }
        return null;
    }
}
