package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.CommonUtils.isEmptyMap;
import static com.fluentretail.se.plugins.util.CommonUtils.isEmptyOrBlank;
import static com.fluentretail.se.plugins.util.Constants.ATTR_SUBSTITUTION_PRICE;
import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.JSON;
import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.STRING;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_RETURN_DESTINATION_LOCATION;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_TAX_TYPE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.ORDER;
import static com.fluentretail.se.plugins.util.Constants.EventField.EVENT_FIELD_NEW_REVISED_ORDER_ID;
import static com.fluentretail.se.plugins.util.Constants.EventField.EVENT_FIELD_NEW_REVISED_ORDER_LINK;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_EXCHANGE_ORDER_REF;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_REMAIN_REFUND_AMOUNT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_ORDER_REF;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_BALANCE_REFUND_AMOUNT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_ORIGINAL_REFUND_AMOUNT;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_PRICE;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_TAX_PRICE;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_TOTAL_PRICE;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_TOTAL_TAX_PRICE;
import static com.fluentretail.se.plugins.util.OrderUtils.generateUniqueOrderRefSuffix;
import static java.lang.String.format;
import static java.lang.String.join;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.util.ISO8601DateFormat;
import com.fluentcommerce.model.attribute.Attribute;
import com.fluentcommerce.se.common.graphql.mutations.order.CreateOrderMutation;
import com.fluentcommerce.se.common.graphql.mutations.order.CreateReturnOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByIdBaseQuery;
import com.fluentcommerce.se.common.graphql.queries.setting.GetSettingsByNameQuery;
import com.fluentcommerce.se.common.graphql.type.AmountTypeInput;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.CreateCustomerAddressInput;
import com.fluentcommerce.se.common.graphql.type.CreateFinancialTransactionWithOrderInput;
import com.fluentcommerce.se.common.graphql.type.CreateFulfilmentChoiceWithOrderInput;
import com.fluentcommerce.se.common.graphql.type.CreateOrderInput;
import com.fluentcommerce.se.common.graphql.type.CreateOrderItemWithOrderInput;
import com.fluentcommerce.se.common.graphql.type.CreateReturnOrderItemWithReturnOrderInput;
import com.fluentcommerce.se.common.graphql.type.CreateReturnVerificationWithReturnOrderInput;
import com.fluentcommerce.se.common.graphql.type.CurrencyKey;
import com.fluentcommerce.se.common.graphql.type.CustomerId;
import com.fluentcommerce.se.common.graphql.type.CustomerKey;
import com.fluentcommerce.se.common.graphql.type.LocationLinkInput;
import com.fluentcommerce.se.common.graphql.type.OrderItemLinkInput;
import com.fluentcommerce.se.common.graphql.type.OrderLinkInput;
import com.fluentcommerce.se.common.graphql.type.ProductCatalogueKey;
import com.fluentcommerce.se.common.graphql.type.ProductKey;
import com.fluentcommerce.se.common.graphql.type.QuantityTypeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.SettingValueTypeInput;
import com.fluentcommerce.se.common.graphql.type.StreetAddressInput;
import com.fluentcommerce.se.common.graphql.type.TaxTypeInput;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RubixException;
import com.fluentretail.rubix.rule.meta.EventAttribute;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.EventInfoVariables;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.util.ValueConverter;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.helper.oms.GqlQueryHelper;
import com.fluentretail.se.plugins.util.AttributeUtils;
import com.fluentretail.se.plugins.util.helpers.TaxHelper;
import com.google.common.collect.ImmutableList;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import lombok.Builder;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@RuleInfo(
    name = "CreateReturnOrderFromOrder",
    description = "Creates a return entity to begin the process of refunding the returned items.",
    accepts = {
        @EventInfo(entityType = ORDER)
    },
    produces = {
        @EventInfo(
            entityType = EventInfoVariables.EVENT_TYPE,
            entitySubtype = EventInfoVariables.EVENT_SUBTYPE,
            status = EventInfoVariables.EVENT_STATUS
        )
    }
)
@EventAttribute(name = "returnItems",
    type = "RETURN_ITEMS")
@EventAttribute(name = "exchangeItems")
@EventAttribute(name = "pickupLocation")
@EventAttribute(name = "lodgedLocation")
@EventAttribute(name = "destinationLocation")
@EventAttribute(name = "type")
@Slf4j
public class CreateReturnOrderFromOrder implements Rule {

    public static final String DEFAULT_RETURN_ITEM_TYPE = "RETURN_ITEM";
    private static final String CLASS_NAME = CreateReturnOrderFromOrder.class.getSimpleName();
    private static final ISO8601DateFormat DATE_TIME_FORMATTER = new ISO8601DateFormat();

    /**
     * Try get the product ref from within an OrderItem.
     */
    public static String getProductRef(String rootPath, GetOrderByIdBaseQuery.Product product) {
        if (product == null) {
            throw new IllegalArgumentException(
                format("Corresponding OrderItem for %s contains missing product", rootPath));
        }
        try {
            if (product.asStandardProduct() != null) {
                return product.asStandardProduct().ref();
            }
            if (product.asGroupProduct() != null) {
                return product.asGroupProduct().ref();
            }
            if (product.asVariantProduct() != null) {
                return product.asVariantProduct().ref();
            }
        } catch (Exception ignored) {
        }

        throw new IllegalArgumentException(format("Error retrieving OrderItem.product.ref for %s", rootPath));
    }

    /**
     * Try get the product catalogue ref from within an OrderItem.
     */
    public static String getCatalogueRef(String rootPath, GetOrderByIdBaseQuery.Product product) {
        if (product == null) {
            throw new IllegalArgumentException(
                format("Corresponding OrderItem for %s contains missing product", rootPath));
        }
        try {
            if (product.asStandardProduct() != null) {
                return product.asStandardProduct().catalogue().ref();
            }
            if (product.asGroupProduct() != null) {
                return product.asGroupProduct().catalogue().ref();
            }
            if (product.asVariantProduct() != null) {
                return product.asVariantProduct().catalogue().ref();
            }
        } catch (Exception ignored) {
        }

        throw new IllegalArgumentException(format("Error retrieving OrderItem.product.catalogue.ref for %s", rootPath));
    }

    public static <T> T tryCastNonNull(Object o, Class<T> clazz, String path) {
        if (o == null) {
            return null;
        }
        if (clazz.isInstance(o)) {
            return clazz.cast(o);
        } else if (clazz.isAssignableFrom(Double.class) && o instanceof Integer) {
            return (T) new Double(Integer.class.cast(o));
        }
        throw new IllegalArgumentException(
            format("Expect '%s' to be type '%s' but got '%s'", path, clazz.getSimpleName(),
                o.getClass().getSimpleName()));
    }

    /**
     * Try casting the associated key value to the {@code castTo} type, otherwise throw an
     * {@code IllegalArgumentException}. {@code null} is returned if no key exists.
     */
    @Nullable
    public static <T> T tryGetValueForKeyOrNull(Map<String, Object> payload, String key, String rootPath,
        Class<T> castTo) {
        String path = StringUtils.isEmpty(rootPath) ? key : (rootPath + "." + key);
        return tryCastNonNull(payload.get(key), castTo, path);
    }

    /**
     * Try casting the associated key value to the {@code castTo} type, otherwise throw an
     * {@code IllegalArgumentException} if the value is null or there is a cast error.
     */
    @Nonnull
    public static <T> T tryGetMandatoryValueForKeyOrThrow(Map<String, Object> payload, String key, String rootPath,
        Class<T> castTo) {
        String path = StringUtils.isEmpty(rootPath) ? key : (rootPath + "." + key);
        T value = tryCastNonNull(payload.get(key), castTo, path);
        if (value == null) {
            throw new IllegalArgumentException(format("Required attribute at path '%s' is missing", path));
        }
        return value;
    }

    /**
     * Try casting the associated key value to a String otherwise an {@code IllegalArgumentException} is thrown if no
     * key exists or the value is null.
     */
    public static String tryGetMandatoryStringOrThrow(Map<String, Object> payload, String key, String rootPath) {
        return tryGetMandatoryValueForKeyOrThrow(payload, key, rootPath, String.class);
    }

    /**
     * Try casting the associated key value to a String otherwise an {@code IllegalArgumentException} is thrown if no
     * key exists or the value is empty or blank.
     */
    public static String tryGetMandatoryValidatedStringOrThrow(Map<String, Object> payload, String key,
        String rootPath) {
        String path = StringUtils.isEmpty(rootPath) ? key : (rootPath + "." + key);
        String value = tryGetMandatoryValueForKeyOrThrow(payload, key, rootPath, String.class);

        if (isEmptyOrBlank(value)) {
            throw new IllegalArgumentException(
                format("Required attribute at path '%s' cannot contain empty or blank string", path));
        }
        return value;
    }

    /**
     * Try casting the associated key value to a Double otherwise an {@code IllegalArgumentException} is thrown if no
     * key exists or the value is negative.
     */
    public static Double tryGetMoneyValue(Map<String, Object> payload, String key, String rootPath) {
        String path = StringUtils.isEmpty(rootPath) ? key : (rootPath + "." + key);
        Double value = tryCastNonNull(payload.get(key), Double.class, path);

        if (value != null && value < 0) {
            throw new IllegalArgumentException(format("Money amount at path '%s' cannot be negative", path));
        }
        return value;
    }

    @Override
    public void run(Context context) {
        Event event = context.getEvent();
        String orderId = event.getEntityId();

        try {
            log.info("[{}] - rule: {} incoming event {}", event.getAccountId(), CLASS_NAME, event);
            validateBasicContext(context);

            GqlQueryHelper gqlQueryHelper = new GqlQueryHelper(context);

            GetOrderByIdBaseQuery.Data getOrderByIdData = gqlQueryHelper.getOrderById(orderId, null, true,
                true, true, false, true, true);
            validateOrderExists(getOrderByIdData, orderId);

            Map<String, GetOrderByIdBaseQuery.Node> allOrderItems = gqlQueryHelper.getAllOrderItemsByOrderItemRef(
                getOrderByIdData);
            CreateReturnOrderSettings settings = getSettings(event, gqlQueryHelper);

            final String returnOrderRef = generateReturnOrderRef(event, getOrderByIdData.orderById().ref());
            final String exchangeOrderRef = generateExchangeOrderRef(event, getOrderByIdData.orderById().ref());
            Map<String, Object> exchange = tryGetValueForKeyOrNull(event.getAttributes(), "exchangeItems", "",
                Map.class);
            boolean evenExchange;
            ArrayList exchangeItems = null;
            ArrayList financialTransactions = null;
            double remainRefundAmount = 0.0;
            double exchangeOrderPriceCorrection = 1.0;
            double returnOrderPriceCorrection = 1.0;

            Map<String, Object> refundData = new HashMap<>();

            if (exchange != null) {
                evenExchange = tryGetMandatoryValueForKeyOrThrow(exchange, "even", "", Boolean.class);
                exchangeItems = tryGetValueForKeyOrNull(exchange, "items", "", ArrayList.class);
                financialTransactions = tryGetValueForKeyOrNull(exchange, "financialTransactions", "", ArrayList.class);

                if (!evenExchange) {
                    ArrayList returnItems = tryGetValueForKeyOrNull(event.getAttributes(), "returnItems", "",
                        ArrayList.class);
                    if (returnItems == null || returnItems.isEmpty()) {
                        throw new IllegalArgumentException("returnItems is missing or contains 0 return items");
                    }

                    double returnOrderTotal = calculateReturnOrderTotal(context, returnItems, allOrderItems);
                    double exchangeOrderTotal = calculateExchangeOrderTotal(context, exchangeItems);
                    log.info(format("Uneven exchange! Return Order total: %.2f, Exchange Order total: %.2f",
                        returnOrderTotal, exchangeOrderTotal));

                    if (returnOrderTotal > exchangeOrderTotal) {
                        remainRefundAmount = returnOrderTotal - exchangeOrderTotal;
                        returnOrderPriceCorrection -= exchangeOrderTotal / returnOrderTotal;
                        refundData.put(PROP_ORIGINAL_REFUND_AMOUNT, returnOrderTotal);
                        refundData.put(PROP_BALANCE_REFUND_AMOUNT, remainRefundAmount);
                    } else if (returnOrderTotal < exchangeOrderTotal) {
                        exchangeOrderPriceCorrection -= returnOrderTotal / exchangeOrderTotal;
                        returnOrderPriceCorrection = 0.0;
                    } else {
                        evenExchange = true;
                    }
                }
            }
            boolean isExchange = exchangeItems != null && exchangeItems.size() > 0;

            CreateReturnOrderMutation.Builder builder = CreateReturnOrderMutation.builder();

            addRetailerId(builder, event);
            builder.ref(returnOrderRef);
            addType(builder, event);
            addCustomerRef(builder, getOrderByIdData);
            addOrderLink(builder, event, getOrderByIdData);
            addLocations(builder, event, settings);
            addCurrency(builder, event, getOrderByIdData);
            addDefaultTaxType(builder, event, settings);
            addReturnMetaData(builder, event);
            addReturnItemsWithAggregateTotals(builder, event, settings, getOrderByIdData, allOrderItems,
                returnOrderPriceCorrection);
            if (isExchange) {
                builder.exchangeOrder(OrderLinkInput.builder()
                    .ref(exchangeOrderRef)
                    .retailer(RetailerId.builder().id(event.getRetailerId()).build())
                    .build());
                if (remainRefundAmount > 0.0) {
                    builder.attributes(Arrays.asList(
                        AttributeInput.builder()
                            .name(ATTR_EXCHANGE_ORDER_REF)
                            .type(STRING)
                            .value(exchangeOrderRef)
                            .build(),
                        AttributeInput.builder()
                            .name(ATTR_REMAIN_REFUND_AMOUNT)
                            .type(JSON)
                            .value(refundData)
                            .build()
                    ));
                } else {
                    builder.attributes(Collections.singletonList(AttributeInput.builder()
                        .name(ATTR_EXCHANGE_ORDER_REF)
                        .type(STRING)
                        .value(exchangeOrderRef)
                        .build()));
                }
            }

            context.action().mutation(builder.build());

            if (isExchange) {
                createExchangeOrder(context, event, getOrderByIdData.orderById(), exchangeItems, exchangeOrderRef,
                    returnOrderRef, financialTransactions, exchangeOrderPriceCorrection);
            }
        } catch (Exception ex) {
            log.error("[{}] - rule: {} exception message: {}, stackTrace: {}",
                event.getAccountId(), CLASS_NAME,
                ex.getMessage() == null ? ex.toString() : ex.getMessage(), Arrays.toString(ex.getStackTrace()));
            throw new RubixException(422,
                format("Rule: " + CLASS_NAME + " - exception message: %s. Stacktrace: %s", ex.getMessage(),
                    Arrays.toString(ex.getStackTrace())));
        }
    }

    private Object getNodeValue(GetSettingsByNameQuery.Node node) {
        switch (node.valueType()) {
            case "JSON":
            case "LOB":
                return node.lobValue();
            default:
                return node.value();
        }
    }

    /**
     * Return a simple type containing settings to use as fallback values should they be missing in the event.
     */
    private CreateReturnOrderSettings getSettings(Event event, GqlQueryHelper gqlQueryHelper) {
        GetSettingsByNameQuery query = gqlQueryHelper.buildSettingsByNameQuery(
            ImmutableList.of(DEFAULT_TAX_TYPE, DEFAULT_RETURN_DESTINATION_LOCATION),
            ImmutableList.of("RETAILER"),
            ImmutableList.of(Integer.parseInt(event.getRetailerId())),
            null
        );

        GetSettingsByNameQuery.Data data = gqlQueryHelper.runSettingsByNameQuery(query);
        Map<String, Object> settingsMap = gqlQueryHelper.getAllSettingsByNameQuery(query.variables(), data).stream()
            .filter(node -> node != null && getNodeValue(node) != null)
            .collect(Collectors.toMap(GetSettingsByNameQuery.Node::name, this::getNodeValue));

        // find the destinationLocation from the event first, else try the settings fallback
        String destinationLocation = tryGetValueForKeyOrNull(event.getAttributes(), "destinationLocation", "",
            String.class);
        if (isEmptyOrBlank(destinationLocation)) {
            destinationLocation = tryGetValueForKeyOrNull(settingsMap, DEFAULT_RETURN_DESTINATION_LOCATION, "",
                String.class);
        }
        if (isEmptyOrBlank(destinationLocation)) {
            throw new IllegalArgumentException(format(
                "setting '%s' is missing, please provide a valid 'destinationLocation' in the event attributes or configure the setting.",
                DEFAULT_RETURN_DESTINATION_LOCATION));
        }

        TaxTypeInput taxTypeInput = null;

        if (settingsMap.containsKey(DEFAULT_TAX_TYPE)) {
            try {
                ObjectNode lob = (ObjectNode) settingsMap.get(DEFAULT_TAX_TYPE);
                String country = lob.get("country").asText();
                String group = lob.get("group").asText();
                String tariff = lob.get("tariff").asText();

                if (isEmptyOrBlank(country) || isEmptyOrBlank(group) || isEmptyOrBlank(tariff)) {
                    throw new IllegalArgumentException(
                        "country, group and tariff values must all have at least 1 character");
                }

                taxTypeInput = TaxTypeInput.builder()
                    .country(country)
                    .group(group)
                    .tariff(tariff)
                    .build();

            } catch (Exception e) {
                throw new IllegalArgumentException(
                    format(
                        "setting '%s' incorrectly configured, expect LOB value with 3 keys - country, group, tariff. exception message: %s",
                        DEFAULT_TAX_TYPE,
                        e.getMessage())
                );
            }
        }

        return CreateReturnOrderSettings.builder()
            .defaultTaxType(taxTypeInput)
            .destinationLocation(destinationLocation)
            .build();
    }

    private void validateOrderExists(GetOrderByIdBaseQuery.Data data, String orderId) {
        GetOrderByIdBaseQuery.OrderById order = data.orderById();

        if (order == null) {
            throw new IllegalArgumentException(format("Order was not found by id '%s'", orderId));
        }
    }

    private void addCustomerRef(CreateReturnOrderMutation.Builder builder, GetOrderByIdBaseQuery.Data data) {
        GetOrderByIdBaseQuery.OrderById order = data.orderById();
        GetOrderByIdBaseQuery.Customer customer = order.customer();
        if (customer == null) {
            throw new IllegalArgumentException(
                format("Order id '%s' has no Customer which is required to create a valid ReturnOrder entity",
                    order.id()));
        }

        String customerRefFromOrder = customer.ref();
        if (StringUtils.isEmpty(customerRefFromOrder)) {
            throw new IllegalArgumentException(
                format("Order id '%s' has no Customer.ref which is required to create a valid ReturnOrder entity",
                    order.id()));
        }

        builder.customer(CustomerKey.builder().ref(customerRefFromOrder).build());
    }

    private void addRetailerId(CreateReturnOrderMutation.Builder builder, Event event) {
        builder.retailer(RetailerId.builder().id(event.getRetailerId()).build());
    }

    private AmountTypeInput createAmountType(Map<String, Object> amountTypeMap, String rootPath) {
        if (!amountTypeMap.containsKey("amount")) {
            throw new IllegalArgumentException(format("amount key is missing from '%s'", rootPath));
        }

        return AmountTypeInput.builder().amount(tryGetMoneyValue(amountTypeMap, "amount", rootPath)).build();
    }

    /**
     * Add return order meta data fields if they exist - returnVerifications, returnAuthorisationKey,
     * returnAuthorisationKeyExpiry, returnAuthorisationDisposition
     */
    private void addReturnMetaData(CreateReturnOrderMutation.Builder builder, Event event) {
        addReturnVerifications(builder, event);
        addReturnAuthorisationDisposition(builder, event);

        builder.returnAuthorisationKey(
            tryGetValueForKeyOrNull(event.getAttributes(), "returnAuthorisationKey", "", String.class));

        String returnAuthorisationKeyExpiry = tryGetValueForKeyOrNull(event.getAttributes(),
            "returnAuthorisationKeyExpiry", "", String.class);
        if (!isEmptyOrBlank(returnAuthorisationKeyExpiry)) {
            try {
                Date parsedKeyExpiryDate = DATE_TIME_FORMATTER.parse(returnAuthorisationKeyExpiry);
                builder.returnAuthorisationKeyExpiry(parsedKeyExpiryDate);
            } catch (Exception e) {
                throw new IllegalArgumentException(
                    "returnAuthorisationKeyExpiry is not in UTC date format, must comply with ISO8601");
            }
        }
    }

    private SettingValueTypeInput createReturnItemSettingValue(Map<String, Object> returnItemMap, String key,
        String rootPath) {
        Map maybeSettingsMap = tryGetValueForKeyOrNull(returnItemMap, key, rootPath, Map.class);
        if (isEmptyMap(maybeSettingsMap)) {
            return null;
        }

        String path = StringUtils.isEmpty(rootPath) ? key : (rootPath + "." + key);
        Map<String, Object> settingsMap = (Map<String, Object>) maybeSettingsMap;

        SettingValueTypeInput.Builder builder = SettingValueTypeInput.builder();
        builder.value(tryGetMandatoryStringOrThrow(settingsMap, "value", path));
        builder.label(tryGetValueForKeyOrNull(settingsMap, "label", path, String.class));

        return builder.build();
    }

    private void addOrderLink(CreateReturnOrderMutation.Builder builder, Event event, GetOrderByIdBaseQuery.Data data) {
        String orderRef = data.orderById().ref();

        if (StringUtils.isEmpty(orderRef)) {
            // Order.ref will always be available as its mandatory for a createOrder gql mutation.
            throw new IllegalArgumentException("Order.ref is empty");
        }

        builder.order(OrderLinkInput.builder()
            .ref(orderRef)
            .retailer(RetailerId.builder().id(event.getRetailerId()).build())
            .build());
    }

    /**
     * type is mandatory
     */
    private void addType(CreateReturnOrderMutation.Builder builder, Event event) {
        builder.type(tryGetMandatoryValidatedStringOrThrow(event.getAttributes(), "type", ""));
    }

    /**
     * Since Order does not store currency, by convention the first OrderItem.currency is used. It's assumed all items
     * will be using the same currency. However, if the event contains a currency, its alphabeticCode must contain at
     * least 1 character otherwise the currency in the first order item will be used.
     */
    private void addCurrency(CreateReturnOrderMutation.Builder builder, Event event,
        GetOrderByIdBaseQuery.Data GetOrderByIdBaseQuery) {
        Map maybeCurrencyMap = tryGetValueForKeyOrNull(event.getAttributes(), "currency", "", Map.class);
        if (!isEmptyMap(maybeCurrencyMap)) {
            String alphabeticCode = tryGetValueForKeyOrNull(((Map<String, Object>) maybeCurrencyMap), "alphabeticCode",
                "currency", String.class);

            if (!isEmptyOrBlank(alphabeticCode)) {
                builder.currency(CurrencyKey.builder().alphabeticCode(alphabeticCode).build());
                return;
            }
        }

        String currency = null;
        try {
            currency = GetOrderByIdBaseQuery.orderById().items().edges().get(0).node().currency();
        } catch (Exception ignored) {
        }
        if (isEmptyOrBlank(currency)) {
            throw new IllegalArgumentException(
                "No currency provided in event and unable to resolve a valid currency from the first order item");
        }
        builder.currency(CurrencyKey.builder().alphabeticCode(currency).build());
    }

    /**
     * returnVerifications is optional, but if provided all 3 keys are mandatory.
     */
    private void addReturnVerifications(CreateReturnOrderMutation.Builder builder, Event event) {
        CreateReturnVerificationWithReturnOrderInput.Builder verificationBuilder =
            CreateReturnVerificationWithReturnOrderInput.builder();

        Map<String, Object> verificationsMap = (Map<String, Object>) event.getAttribute("returnVerifications",
            Map.class);
        if (isEmptyMap(verificationsMap)) {
            return;
        }

        verificationBuilder.ref(tryGetMandatoryStringOrThrow(verificationsMap, "ref", "returnVerifications"));
        verificationBuilder.type(tryGetMandatoryStringOrThrow(verificationsMap, "type", "returnVerifications"));
        verificationBuilder.verificationDetails(
            tryGetMandatoryStringOrThrow(verificationsMap, "verificationDetails", "returnVerifications"));

        builder.returnVerifications(ImmutableList.of(verificationBuilder.build()));
    }

    /**
     * returnAuthorisationDisposition is optional, but if provided only the 'value' key is mandatory.
     */
    private void addReturnAuthorisationDisposition(CreateReturnOrderMutation.Builder builder, Event event) {
        SettingValueTypeInput.Builder settingBuilder = SettingValueTypeInput.builder();

        Map<String, Object> dispositionMap = (Map<String, Object>) event.getAttribute("returnAuthorisationDisposition",
            Map.class);
        if (isEmptyMap(dispositionMap)) {
            return;
        }

        settingBuilder.value(tryGetMandatoryStringOrThrow(dispositionMap, "value", "returnAuthorisationDisposition"));
        settingBuilder.label(
            tryGetValueForKeyOrNull(dispositionMap, "label", "returnAuthorisationDisposition", String.class));

        builder.returnAuthorisationDisposition(settingBuilder.build());
    }

    /**
     * Either a {@code pickupLocation} or {@code lodgedLocation} is supplied, not both. No locations are validated.
     * {@code destinationLocation} is resolved from the event, then try use a default setting otherwise the rule fails
     * with an exception.
     */
    private void addLocations(CreateReturnOrderMutation.Builder builder, Event event,
        CreateReturnOrderSettings settings) {
        String destinationLocation = tryGetValueForKeyOrNull(event.getAttributes(), "destinationLocation", "",
            String.class);
        if (isEmptyOrBlank(destinationLocation)) {
            destinationLocation = settings.getDestinationLocation();
        }
        builder.destinationLocation(LocationLinkInput.builder().ref(destinationLocation).build());

        boolean hasPickupAddress = false;
        boolean hasLodgedLocation = false;
        String lodgedLocation = null;

        Map maybePickupLocation = tryGetValueForKeyOrNull(event.getAttributes(), "pickupLocation", "", Map.class);
        if (!isEmptyMap(maybePickupLocation)) {
            Map<String, Object> pickupLocationMap = (Map<String, Object>) maybePickupLocation;

            lodgedLocation =
                tryGetValueForKeyOrNull(pickupLocationMap, "lodgedLocation", "pickupLocation", String.class);
            if (!isEmptyOrBlank(lodgedLocation)) {
                hasLodgedLocation = true;
            }

            String street = tryGetValueForKeyOrNull(pickupLocationMap, "street", "pickupLocation", String.class);
            if (!isEmptyOrBlank(street)) {
                hasPickupAddress = true;
                builder.pickupAddress(StreetAddressInput.builder()
                    .companyName(
                        tryGetValueForKeyOrNull(pickupLocationMap, "companyName", "pickupLocation", String.class))
                    .name(tryGetValueForKeyOrNull(pickupLocationMap, "name", "pickupLocation", String.class))
                    .street(street)
                    .city(tryGetValueForKeyOrNull(pickupLocationMap, "city", "pickupLocation", String.class))
                    .state(tryGetValueForKeyOrNull(pickupLocationMap, "state", "pickupLocation", String.class))
                    .postcode(tryGetValueForKeyOrNull(pickupLocationMap, "postcode", "pickupLocation", String.class))
                    .region(tryGetValueForKeyOrNull(pickupLocationMap, "region", "pickupLocation", String.class))
                    .country(tryGetValueForKeyOrNull(pickupLocationMap, "country", "pickupLocation", String.class))
                    .latitude(tryGetValueForKeyOrNull(pickupLocationMap, "latitude", "pickupLocation", Double.class))
                    .longitude(tryGetValueForKeyOrNull(pickupLocationMap, "longitude", "pickupLocation", Double.class))
                    .timeZone(tryGetValueForKeyOrNull(pickupLocationMap, "timeZone", "pickupLocation", String.class))
                    .build());
            }
        }

        if (!hasLodgedLocation) {
            lodgedLocation = tryGetValueForKeyOrNull(event.getAttributes(), "lodgedLocation", "", String.class);
            hasLodgedLocation = !isEmptyOrBlank(lodgedLocation);
        }

        if (hasPickupAddress && hasLodgedLocation) {
            throw new IllegalArgumentException("Either supply pickupLocation or lodgedLocation, not both");
        }
        if (!hasPickupAddress && !hasLodgedLocation) {
            throw new IllegalArgumentException("Either supply pickupLocation or lodgedLocation, both are missing");
        }
        if (hasPickupAddress) {
            return;
        }

        // implies !hasPickupAddress && hasLodgedLocation
        builder.lodgedLocation(LocationLinkInput.builder().ref(lodgedLocation).build());
    }

    /**
     * event.taxType maps to GQL mutation defaultTaxType
     * <ol>
     * <li>Try get taxType from the event</li>
     * <li>Try lookup the default tax type from settings</li>
     * <li>Throw exception since this field is mandatory in gql mutation</li>
     * </ol>
     */
    private void addDefaultTaxType(CreateReturnOrderMutation.Builder builder, Event event,
        CreateReturnOrderSettings settings) {
        Map maybeTaxType = tryGetValueForKeyOrNull(event.getAttributes(), "taxType", "", Map.class);
        if (!isEmptyMap(maybeTaxType)) {
            Map<String, Object> taxTypeMap = (Map<String, Object>) maybeTaxType;
            String country = tryGetValueForKeyOrNull(taxTypeMap, "country", "taxType", String.class);
            String group = tryGetValueForKeyOrNull(taxTypeMap, "group", "taxType", String.class);
            String tariff = tryGetValueForKeyOrNull(taxTypeMap, "tariff", "taxType", String.class);

            if (!isEmptyOrBlank(country) && !isEmptyOrBlank(group) && !isEmptyOrBlank(tariff)) {
                builder.defaultTaxType(
                    TaxTypeInput.builder()
                        .country(country)
                        .group(group)
                        .tariff(tariff)
                        .build()
                );
                return;
            }
        }

        if (settings.getDefaultTaxType() == null) {
            throw new IllegalArgumentException(format(
                "Invalid or missing 'taxType' attribute in ReturnOrder event and there is no fallback setting '%s'",
                DEFAULT_TAX_TYPE));
        }

        builder.defaultTaxType(settings.getDefaultTaxType());
    }

    /**
     * Either supply all 3 aggregate totals (not some random combination), otherwise if none are supplied it will
     * trigger the return item calculation
     */
    private boolean addReturnOrderAggregateAmounts(CreateReturnOrderMutation.Builder builder, Event event) {
        Map subTotalAmountMap = tryGetValueForKeyOrNull(event.getAttributes(), "subTotalAmount", "", Map.class);
        Map totalTaxMap = tryGetValueForKeyOrNull(event.getAttributes(), "totalTax", "", Map.class);
        Map totalAmountMap = tryGetValueForKeyOrNull(event.getAttributes(), "totalAmount", "", Map.class);

        List<String> foundValidAmount = new ArrayList<>();
        if (!isEmptyMap(subTotalAmountMap)) {
            foundValidAmount.add("subTotalAmount");
        }
        if (!isEmptyMap(totalTaxMap)) {
            foundValidAmount.add("totalTax");
        }
        if (!isEmptyMap(totalAmountMap)) {
            foundValidAmount.add("totalAmount");
        }

        if (foundValidAmount.isEmpty()) {
            // Calculate aggregate totals based on Return OrderItem(s).
            return false;
        }

        if (foundValidAmount.size() != 3) {
            throw new IllegalArgumentException(format(
                "Detected missing aggregate totals, expect all 3 attributes to be provided (subTotalAmount, totalTax, totalAmount) but "
                    + "got (%s)",
                join(", ", foundValidAmount)));
        }

        builder.subTotalAmount(createAmountType((Map<String, Object>) subTotalAmountMap, "subTotalAmount"));
        builder.totalTax(createAmountType((Map<String, Object>) totalTaxMap, "totalTax"));
        builder.totalAmount(createAmountType((Map<String, Object>) totalAmountMap, "totalAmount"));

        return true;
    }

    private Double calculateReturnItemUnitAmount(Map<String, Object> returnItemMap,
        GetOrderByIdBaseQuery.Node orderItem, String rootPath, double priceCorrection) {
        Map maybeUnitAmountMap = tryGetValueForKeyOrNull(returnItemMap, "unitAmount", rootPath, Map.class);
        if (!isEmptyMap(maybeUnitAmountMap)) {
            Map<String, Object> unitAmountMap = (Map<String, Object>) maybeUnitAmountMap;
            Double unitAmount = tryGetMoneyValue(unitAmountMap, "amount", rootPath + ".unitAmount");
            if (unitAmount != null) {
                return correctAmountValue(unitAmount, priceCorrection);
            }
        }

        Double price = orderItem.price();
        if (price != null && price > 0) {
            return correctAmountValue(price, priceCorrection);
        }
        price = getPriceForSubstituteItem(orderItem, SUBSTITUTION_PRICE);
        if (price != null && price > 0) {
            return correctAmountValue(price, priceCorrection);
        }

        Double totalPrice = orderItem.totalPrice();
        if (totalPrice != null && totalPrice > 0) {
            return correctAmountValue(totalPrice / orderItem.quantity(), priceCorrection);
        }
        totalPrice = getPriceForSubstituteItem(orderItem, SUBSTITUTION_TOTAL_PRICE);
        if (totalPrice != null && totalPrice > 0) {
            return correctAmountValue(totalPrice / orderItem.quantity(), priceCorrection);
        }

        throw new IllegalArgumentException(
            format("%s cannot be calculated due to missing pricing information in the corresponding OrderItem, "
                + "please supply a unitAmount in the event", rootPath + ".unitAmount.amount"));
    }

    private Double calculateReturnItemTaxAmount(Map<String, Object> returnItemMap, GetOrderByIdBaseQuery.Node orderItem,
        int returnItemQuantity, String rootPath, double priceCorrection) {
        Map maybeItemTaxAmountMap = tryGetValueForKeyOrNull(returnItemMap, "itemTaxAmount", rootPath, Map.class);
        if (!isEmptyMap(maybeItemTaxAmountMap)) {
            Map<String, Object> itemTaxAmountMap = (Map<String, Object>) maybeItemTaxAmountMap;
            Double unitTaxAmount = tryGetMoneyValue(itemTaxAmountMap, "amount", rootPath + ".itemTaxAmount");
            if (unitTaxAmount != null) {
                return correctAmountValue(unitTaxAmount, priceCorrection);
            }
        }

        Double taxPrice = orderItem.taxPrice();
        if (taxPrice != null && taxPrice > 0) {
            return correctAmountValue(taxPrice, priceCorrection);
        }
        taxPrice = getPriceForSubstituteItem(orderItem, SUBSTITUTION_TAX_PRICE);
        if (taxPrice != null && taxPrice > 0) {
            return correctAmountValue(taxPrice, priceCorrection);
        }

        Double totalTaxPrice = orderItem.totalTaxPrice();
        if (totalTaxPrice != null && totalTaxPrice > 0) {
            return correctAmountValue(totalTaxPrice / orderItem.quantity(), priceCorrection) * returnItemQuantity;
        }
        totalTaxPrice = getPriceForSubstituteItem(orderItem, SUBSTITUTION_TOTAL_TAX_PRICE);
        if (totalTaxPrice != null && totalTaxPrice > 0) {
            return correctAmountValue(totalTaxPrice / orderItem.quantity(), priceCorrection) * returnItemQuantity;
        }

        return 0.0;
    }

    /**
     * use itemAmount from the event otherwise multiply unitAmount by unitQuantity which are both derived values.
     */
    private Double calculateReturnItemAmount(Map<String, Object> returnItemMap, int unitQuantity, double unitAmount,
        String rootPath, double priceCorrection) {
        Map maybeItemAmountMap = tryGetValueForKeyOrNull(returnItemMap, "itemAmount", rootPath, Map.class);
        if (!isEmptyMap(maybeItemAmountMap)) {
            Map<String, Object> itemAmountMap = (Map<String, Object>) maybeItemAmountMap;
            Double itemAmount = tryGetMoneyValue(itemAmountMap, "amount", rootPath + ".itemAmount");
            if (itemAmount != null) {
                return correctAmountValue(itemAmount, priceCorrection);
            }
        }
        return unitAmount * unitQuantity;
    }

    private double correctAmountValue(double value, double correction) {
        return (correction < 1.0) ? BigDecimal.valueOf(value * correction)
            .setScale(2, RoundingMode.HALF_UP).doubleValue() : value;
    }

    ////@todo half UTILS

    /**
     * quantity is mandatory. no other validation is performed as its assumed to have been done in ValidateReturnQty
     * rule.
     */
    private QuantityTypeInput calculateReturnItemUnitQuantity(Map<String, Object> returnItemMap, String rootPath) {
        Map<String, Object> unitQuantityMap = (Map<String, Object>) tryGetMandatoryValueForKeyOrThrow(returnItemMap,
            "unitQuantity", rootPath, Map.class);
        QuantityTypeInput.Builder quantityTypeBuilder = QuantityTypeInput.builder();
        quantityTypeBuilder.quantity(
            tryGetMandatoryValueForKeyOrThrow(unitQuantityMap, "quantity", rootPath + ".unitQuantity", Integer.class));
        quantityTypeBuilder.unit(
            tryGetValueForKeyOrNull(unitQuantityMap, "unit", rootPath + ".unitQuantity", String.class));
        return quantityTypeBuilder.build();
    }

    /**
     * Within the return item, taxType maps to unitTaxType. Use the event value at the return item level else fallback
     * to default tax type setting.
     */
    private TaxTypeInput getReturnItemTaxType(Map<String, Object> returnItemMap, CreateReturnOrderSettings settings,
        String rootPath) {
        Map maybeTaxType = tryGetValueForKeyOrNull(returnItemMap, "taxType", "", Map.class);
        if (!isEmptyMap(maybeTaxType)) {
            Map<String, Object> taxTypeMap = (Map<String, Object>) maybeTaxType;
            String country = tryGetValueForKeyOrNull(taxTypeMap, "country", "taxType", String.class);
            String group = tryGetValueForKeyOrNull(taxTypeMap, "group", "taxType", String.class);
            String tariff = tryGetValueForKeyOrNull(taxTypeMap, "tariff", "taxType", String.class);

            if (!isEmptyOrBlank(country) && !isEmptyOrBlank(group) && !isEmptyOrBlank(tariff)) {
                return TaxTypeInput.builder().country(country).group(group).tariff(tariff).build();
            }
        }

        if (settings.getDefaultTaxType() == null) {
            throw new IllegalArgumentException(
                format("Invalid or missing '%s' attribute in event and there is no fallback setting '%s'",
                    rootPath + ".taxType",
                    DEFAULT_TAX_TYPE));
        }

        return settings.getDefaultTaxType();
    }

    /**
     * Add the return items to to the main return order. Return order aggregate totals are calculated from the return
     * items should they be missing in the event.
     */
    private void addReturnItemsWithAggregateTotals(
        CreateReturnOrderMutation.Builder builder,
        Event event, CreateReturnOrderSettings settings,
        GetOrderByIdBaseQuery.Data getOrderByIdData,
        Map<String, GetOrderByIdBaseQuery.Node> allOrderItems,
        double priceCorrection
    ) {
        ArrayList maybeRawReturnItems = tryGetValueForKeyOrNull(event.getAttributes(), "returnItems", "",
            ArrayList.class);
        if (maybeRawReturnItems == null || maybeRawReturnItems.isEmpty()) {
            throw new IllegalArgumentException("returnItems is missing or contains 0 return items");
        }
        ArrayList<Map<String, Object>> rawReturnItems = (ArrayList<Map<String, Object>>) maybeRawReturnItems;

        List<CreateReturnOrderItemWithReturnOrderInput> returnItems = new ArrayList<>();
        for (int i = 0; i < rawReturnItems.size(); i++) {
            Map<String, Object> returnItem = rawReturnItems.get(i);
            returnItems.add(createReturnItem(i,
                event,
                settings,
                returnItem,
                getOrderByIdData,
                allOrderItems,
                priceCorrection));
        }

        builder.returnOrderItems(returnItems);

        // The following line evaluates if the incoming event has the 3 values (totalTax, totalAmount and subTotal)
        // if not, proceeds to calculate individually...
        boolean hasAggregateTotalsSupplied = addReturnOrderAggregateAmounts(builder, event);

        if (!hasAggregateTotalsSupplied) {
            double subTotalAmount = 0.0;
            double totalTaxAmount = 0.0;

            for (CreateReturnOrderItemWithReturnOrderInput returnItem : returnItems) {
                subTotalAmount += returnItem.itemAmount().amount();
                totalTaxAmount += returnItem.itemTaxAmount().amount() * returnItem.unitQuantity().quantity();
            }
            builder.subTotalAmount(AmountTypeInput.builder().amount(subTotalAmount).build());
            builder.totalTax(AmountTypeInput.builder().amount(totalTaxAmount).build());
            builder.totalAmount(AmountTypeInput.builder().amount(subTotalAmount + totalTaxAmount).build());
            log.info(String.format(
                "Manual calculation of: subTotalAmount = '%s', totalTaxAmount = '%s' and totalAmount = '%s'"
                , subTotalAmount, totalTaxAmount, subTotalAmount + totalTaxAmount));
        }
    }

    private CreateReturnOrderItemWithReturnOrderInput createReturnItem(
        int index,
        Event event,
        CreateReturnOrderSettings settings,
        Map<String, Object> returnItemMap,
        GetOrderByIdBaseQuery.Data getOrderByIdData,
        Map<String, GetOrderByIdBaseQuery.Node> allOrderItems,
        double priceCorrection
    ) {
        CreateReturnOrderItemWithReturnOrderInput.Builder returnItemBuilder =
            CreateReturnOrderItemWithReturnOrderInput.builder();

        String rootPath = format("%s[%d]", "returnItems", index);

        // orderItemRef is mandatory, its used to lookup the corresponding OrderItem to perform derived subtotal calculations
        String orderItemRef = tryGetMandatoryValidatedStringOrThrow(returnItemMap, "orderItemRef", rootPath);
        GetOrderByIdBaseQuery.Node orderItem = allOrderItems.get(orderItemRef);

        if (orderItem == null) {
            throw new IllegalArgumentException(
                format("%s.orderItemRef '%s' does not exist in any of the original order items", rootPath,
                    orderItemRef));
        }

        returnItemBuilder.orderItem(
            OrderItemLinkInput.builder()
                .ref(orderItemRef)
                .order(OrderLinkInput.builder()
                    .ref(getOrderByIdData.orderById().ref())
                    .retailer(RetailerId.builder().id(event.getRetailerId()).build())
                    .build())
                .build()
        );

        // use ref if provided, else fallback to using the mandatory orderItemRef
        String ref = tryGetValueForKeyOrNull(returnItemMap, "ref", rootPath, String.class);
        returnItemBuilder.ref(isEmptyOrBlank(ref) ? orderItemRef : ref);

        String type = tryGetValueForKeyOrNull(returnItemMap, "type", rootPath, String.class);
        returnItemBuilder.type(isEmptyOrBlank(type) ? DEFAULT_RETURN_ITEM_TYPE : type);

        // Build the ProductKey from the OrderItem.
        returnItemBuilder.product(
            ProductKey.builder()
                .ref(getProductRef(rootPath, orderItem.product()))
                .catalogue(ProductCatalogueKey.builder().ref(getCatalogueRef(rootPath, orderItem.product())).build())
                .build()
        );

        // unitQuantity
        QuantityTypeInput unitQuantity = calculateReturnItemUnitQuantity(returnItemMap, rootPath);
        returnItemBuilder.unitQuantity(unitQuantity);

        // unitAmount
        Double unitAmount = calculateReturnItemUnitAmount(returnItemMap, orderItem, rootPath, priceCorrection);
        returnItemBuilder.unitAmount(AmountTypeInput.builder().amount(unitAmount).build());

        // itemAmount
        Double itemAmount =
            calculateReturnItemAmount(returnItemMap, unitQuantity.quantity(), unitAmount, rootPath, priceCorrection);
        returnItemBuilder.itemAmount(AmountTypeInput.builder().amount(itemAmount).build());

        // itemTaxAmount
        returnItemBuilder.itemTaxAmount(AmountTypeInput.builder()
            .amount(calculateReturnItemTaxAmount(returnItemMap,
                orderItem,
                unitQuantity.quantity(),
                rootPath,
                priceCorrection)).build());

        // unitTaxType
        returnItemBuilder.unitTaxType(getReturnItemTaxType(returnItemMap, settings, rootPath));

        // return item meta data
        returnItemBuilder.returnReasonComment(
            tryGetValueForKeyOrNull(returnItemMap, "returnReasonComment", rootPath, String.class));
        returnItemBuilder.returnConditionComment(
            tryGetValueForKeyOrNull(returnItemMap, "returnConditionComment", rootPath, String.class));

        returnItemBuilder.returnReason(createReturnItemSettingValue(returnItemMap, "returnReason", rootPath));
        returnItemBuilder.returnCondition(createReturnItemSettingValue(returnItemMap, "returnCondition", rootPath));
        returnItemBuilder.returnPaymentAction(
            createReturnItemSettingValue(returnItemMap, "returnPaymentAction", rootPath));
        returnItemBuilder.returnDispositionAction(
            createReturnItemSettingValue(returnItemMap, "returnDispositionAction", rootPath));

        // append attributes
        List attributes = tryGetValueForKeyOrNull(returnItemMap, "attributes", "", List.class);
        if (attributes != null) {
            List<Attribute> attributesConverted = ValueConverter.convertList(attributes, Attribute.class);
            returnItemBuilder.attributes(attributesConverted.stream()
                .map(a -> AttributeInput.builder().name(a.getName()).value(a.getValue()).type(a.getType()).build())
                .collect(Collectors.toList()));
        }

        return returnItemBuilder.build();
    }

    private void validateBasicContext(Context context) {
        Event event = context.getEvent();

        if (isEmptyOrBlank(event.getAccountId())) {
            throw new IllegalArgumentException("accountId is empty");
        }
        if (isEmptyOrBlank(event.getRetailerId())) {
            throw new IllegalArgumentException("retailerId is empty");
        }
        if (isEmptyOrBlank(event.getEntityId())) {
            throw new IllegalArgumentException("entityId is empty");
        }
        if (isEmptyOrBlank(event.getEntityType())) {
            throw new IllegalArgumentException("entityType is empty");
        }
        if (isEmptyOrBlank(event.getEntitySubtype())) {
            throw new IllegalArgumentException("entitySubtype is empty");
        }
        if (isEmptyOrBlank(event.getEntityStatus())) {
            throw new IllegalArgumentException("entityStatus is empty");
        }
        if (isEmptyOrBlank(event.getName())) {
            throw new IllegalArgumentException("event name is empty");
        }
        if (isEmptyMap(event.getAttributes())) {
            throw new IllegalArgumentException("attributes is empty");
        }
    }

    private Double getPriceForSubstituteItem(GetOrderByIdBaseQuery.Node orderItem, String priceKey) {
        Optional<GetOrderByIdBaseQuery.Attribute> price = AttributeUtils.getAttribute(ATTR_SUBSTITUTION_PRICE,
            orderItem.attributes());
        if (price.isPresent()) {
            ObjectMapper mapper = new ObjectMapper();
            Map<String, Double> value = mapper.convertValue(price.get().value(),
                new TypeReference<Map<String, Double>>() {
                });
            return value.getOrDefault(priceKey, null);
        }
        return null;
    }

    private String generateReturnOrderRef(Event event, String orderRef) {
        String ref = tryGetValueForKeyOrNull(event.getAttributes(), "ref", "", String.class);
        if (isEmptyOrBlank(ref)) {
            ref = format("%s-R%s", orderRef, generateUniqueOrderRefSuffix());
        }
        return ref;
    }

    private String generateExchangeOrderRef(Event event, String orderRef) {
        String ref = tryGetValueForKeyOrNull(event.getAttributes(), "exRef", "", String.class);
        if (isEmptyOrBlank(ref)) {
            ref = format("%s-EX%s", orderRef, generateUniqueOrderRefSuffix());
        }
        return ref;
    }

    private void createExchangeOrder(Context context, Event event, GetOrderByIdBaseQuery.OrderById order,
        ArrayList exchangeItems, String exchangeOrderRef, String returnOrderRef, ArrayList financialTransactions,
        double priceCorrection) {
        CreateOrderMutation.Builder builder = CreateOrderMutation.builder();

        TotalPriceInfo priceInfo = new TotalPriceInfo();
        List<CreateOrderItemWithOrderInput> items =
            createExchangeOrderItems(context, exchangeItems, priceInfo, priceCorrection);
        List<CreateFinancialTransactionWithOrderInput> transactions =
            ((financialTransactions == null) || financialTransactions.isEmpty()) ? null :
                ((ArrayList<Map<String, Object>>) financialTransactions).stream().map((tr) -> {

                    // Capture amount here as it's used in two places below
                    double amount = tryGetMandatoryValueForKeyOrThrow(tr, "amount", null, Double.class);

                    // Determine if the method is pay by link
                    String paymentMethod = tryGetMandatoryValueForKeyOrThrow(tr, "paymentMethod", null, String.class);

                    log.info("Checking for PAY_BY_LINK method: " + paymentMethod + paymentMethod.equals("PAY_BY_LINK"));

                    // Create the financial transaction (with Adyen references if available)
                    return CreateFinancialTransactionWithOrderInput.builder()
                        .ref(tryGetMandatoryValueForKeyOrThrow(tr, "ref", null, String.class))
                        .type(tryGetMandatoryValueForKeyOrThrow(tr, "type", null, String.class))
                        .amount(amount)
                        .paymentMethod(paymentMethod)
                        .currency(tryGetMandatoryValueForKeyOrThrow(tr, "currency", null, String.class))
                        .cardType(tryGetValueForKeyOrNull(tr, "cardType", null, String.class))
                        .paymentProvider(tryGetValueForKeyOrNull(tr, "paymentProvider", null, String.class))
                        .externalTransactionId(tryGetValueForKeyOrNull(tr, "externalTransactionId", null, String.class))
                        .externalTransactionCode(tryGetValueForKeyOrNull(tr,
                            "externalTransactionCode",
                            null,
                            String.class))
                        .build();
                }).collect(Collectors.toList());
        builder.input(CreateOrderInput.builder()
            .ref(exchangeOrderRef)
            .type(order.type())
            .retailer(RetailerId.builder().id(event.getRetailerId()).build())
            .customer(CustomerId.builder().id(order.customer().id()).build())
            .fulfilmentChoice(createFulfilmentChoice(order))
            .items(items)
            .totalPrice(priceInfo.totalPrice)
            .totalTaxPrice(priceInfo.totalTaxPrice)
            .financialTransactions(transactions)
            .attributes(Arrays.asList(
                    AttributeInput.builder()
                        .name(ATTR_RETURN_ORDER_REF)
                        .type(STRING)
                        .value(returnOrderRef)
                        .build(),
                    AttributeInput.builder()
                        .name(EVENT_FIELD_NEW_REVISED_ORDER_LINK)
                        .type(STRING)
                        .value(order.ref())
                        .build(),
                    AttributeInput.builder()
                        .name(EVENT_FIELD_NEW_REVISED_ORDER_ID)
                        .type(STRING)
                        .value(order.id())
                        .build()
                )
            )
            .build());

        context.action().mutation(builder.build());
    }

    private List<CreateOrderItemWithOrderInput> createExchangeOrderItems(Context context,
        ArrayList rawExchangeItems, TotalPriceInfo priceInfo, double priceCorrection) {
        List<CreateOrderItemWithOrderInput> items = new ArrayList<>();
        ArrayList<Map<String, Object>> exchangeItems = (ArrayList<Map<String, Object>>) rawExchangeItems;
        for (Map<String, Object> item : exchangeItems) {
            String productRef = tryGetMandatoryValueForKeyOrThrow(item, "ref", null, String.class);
            int quantity = tryGetMandatoryValueForKeyOrThrow(item, "quantity", null, Integer.class);
            double price = Double.parseDouble(tryGetMandatoryValueForKeyOrThrow(item, "price", null, String.class));
            if (priceCorrection < 1.0) {
                price = BigDecimal.valueOf(price * priceCorrection).setScale(2, RoundingMode.HALF_UP).doubleValue();
            }
            String currency = tryGetMandatoryValueForKeyOrThrow(item, "currency", null, String.class);
            String taxType = tryGetMandatoryValueForKeyOrThrow(item, "taxType", null, String.class);
            String catalogueRef = tryGetMandatoryValueForKeyOrThrow(item, "productCatalogueRef", null, String.class);
            double taxPercentage = TaxHelper.getTariffPercentageForTariff(context, taxType);
            double priceWithTax = TaxHelper.calculatePaidPriceFromPrice(price, taxPercentage);
            double taxPrice = priceWithTax - price;
            double totalPrice = price * quantity;
            double totalTaxPrice = taxPrice * quantity;

            priceInfo.totalPrice += price * quantity;
            priceInfo.totalTaxPrice += totalTaxPrice;

            items.add(CreateOrderItemWithOrderInput.builder()
                .ref(productRef)
                .productRef(productRef)
                .productCatalogueRef(catalogueRef)
                .quantity(quantity)
                .currency(currency)
                .taxType(taxType)
                .price(price)
                .paidPrice(price)
                .taxPrice(taxPrice)
                .totalPrice(totalPrice)
                .totalTaxPrice(totalTaxPrice)
                .build());
        }
        return items;
    }

    private CreateFulfilmentChoiceWithOrderInput createFulfilmentChoice(GetOrderByIdBaseQuery.OrderById order) {
        CreateFulfilmentChoiceWithOrderInput.Builder builder = CreateFulfilmentChoiceWithOrderInput.builder();
        builder.deliveryType(order.fulfilmentChoice().deliveryType());
        builder.fulfilmentType(order.fulfilmentChoice().fulfilmentType());
        if (order.fulfilmentChoice().pickupLocationRef() != null) {
            builder.pickupLocationRef(order.fulfilmentChoice().pickupLocationRef());
        }
        if (order.fulfilmentChoice().deliveryAddress() != null) {
            GetOrderByIdBaseQuery.DeliveryAddress address = order.fulfilmentChoice().deliveryAddress();
            builder.deliveryAddress(CreateCustomerAddressInput.builder()
                .ref(UUID.randomUUID().toString())
                .name(address.name())
                .companyName(address.companyName())
                .street(address.street())
                .city(address.city())
                .state(address.state())
                .postcode(address.postcode())
                .region(address.region())
                .country(address.country())
                .longitude(address.longitude())
                .latitude(address.latitude())
                .build());
        }
        return builder.build();
    }

    private Double calculateReturnOrderTotal(Context context, ArrayList returnItems,
        Map<String, GetOrderByIdBaseQuery.Node> orderItems) {
        double sum = 0.0;
        ArrayList<Map<String, Object>> rawReturnItems = (ArrayList<Map<String, Object>>) returnItems;
        for (int i = 0; i < rawReturnItems.size(); i++) {
            Map<String, Object> returnItem = rawReturnItems.get(i);
            String rootPath = format("%s[%d]", "returnItems", i);
            String orderItemRef = tryGetMandatoryValidatedStringOrThrow(returnItem, "orderItemRef", rootPath);
            GetOrderByIdBaseQuery.Node orderItem = orderItems.get(orderItemRef);
            if (orderItem == null) {
                throw new IllegalArgumentException(
                    format("OrderItemRef %s does not exist in any of the original order items", orderItemRef));
            }
            Map<String, Object> unitQuantity = (Map<String, Object>) tryGetMandatoryValueForKeyOrThrow(returnItem,
                "unitQuantity", rootPath, Map.class);
            int quantity =
                tryGetMandatoryValueForKeyOrThrow(unitQuantity, "quantity", rootPath + ".unitQuantity", Integer.class);
            sum += BigDecimal.valueOf(quantity * (orderItem.price() + orderItem.taxPrice()))
                .setScale(2, RoundingMode.HALF_UP).doubleValue();
        }
        return BigDecimal.valueOf(sum).setScale(2, RoundingMode.HALF_UP).doubleValue();
    }

    private double calculateExchangeOrderTotal(Context context, ArrayList exchangeItems) {
        double sum = ((ArrayList<Map<String, Object>>) exchangeItems).stream().mapToDouble(it -> {
            int quantity = tryGetMandatoryValueForKeyOrThrow(it, "quantity", null, Integer.class);
            double price = Double.parseDouble(tryGetMandatoryValueForKeyOrThrow(it, "price", null, String.class));
            String taxType = tryGetMandatoryValueForKeyOrThrow(it, "taxType", null, String.class);
            double taxPercentage = TaxHelper.getTariffPercentageForTariff(context, taxType);
            return BigDecimal.valueOf(TaxHelper.calculatePaidPriceFromPrice(price, taxPercentage) * quantity)
                .setScale(2, RoundingMode.HALF_UP).doubleValue();
        }).sum();
        return BigDecimal.valueOf(sum).setScale(2, RoundingMode.HALF_UP).doubleValue();
    }

    private static class TotalPriceInfo {

        double totalPrice;
        double totalTaxPrice;

        TotalPriceInfo() {
            totalPrice = totalTaxPrice = 0.0;
        }
    }

    @Builder(builderClassName = "Builder",
        toBuilder = true)
    @Getter
    public static class CreateReturnOrderSettings {

        /**
         * Contains the configured DB setting or null if no setting exists and attempt to extract values from the
         * inbound event.
         */
        private TaxTypeInput defaultTaxType;

        /**
         * Contains the value from the inbound event, otherwise the db setting should one be configured. This will never
         * be null to simplify consuming code.
         */
        private String destinationLocation;
    }

}
