package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;

import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.Date;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "ScheduleEventIfRMAExpirationIsSet",
    description = "Schedules event {" + PROP_EVENT_NAME + "} if Return Order has RMA expiration time",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(
    name = PROP_EVENT_NAME,
    description = "Event name to be sent")
@Slf4j
public class ScheduleEventIfRMAExpirationIsSet implements Rule {

    private static final String CLASS_NAME = ScheduleEventIfRMAExpirationIsSet.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME);
        final String eventName = context.getProp(PROP_EVENT_NAME);

        final String returnOrderRef = context.getEntity().getRef();
        GetReturnOrderQuery.Data returnOrder =
            (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .includeAttributes(false)
                .includeReturnOrderItems(false)
                .includePickupLocation(false)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null) {
            String message = MessageFormat.format("{0} - Return Order with ref {1} has not been found!",
                logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        if (returnOrder.returnOrder().returnAuthorisationKeyExpiry() == null) {
            log.warn(MessageFormat.format("{0} - RMA Expiration Date is not set!", logPrefix));
            return;
        }
        Date expirationTime;
        try {
            expirationTime = (Date) returnOrder.returnOrder().returnAuthorisationKeyExpiry();
        } catch (Exception ex) {
            log.warn(MessageFormat.format("{0} - Bad value of RMA Expiration Date: {1}",
                logPrefix, returnOrder.returnOrder().returnAuthorisationKeyExpiry().toString()));
            return;
        }
        if (expirationTime.after(new Date())) {
            Event event = context.getEvent().toBuilder().name(eventName).scheduledOn(expirationTime).build();
            log.info(MessageFormat.format("{0} - Sending event: {1}", logPrefix, event));
            context.action().sendEvent(event);
        }
    }
}
