package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RubixException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.EventInfoVariables;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.Constants.EntityType;
import com.fluentretail.se.plugins.util.RuleUtils;
import com.google.common.collect.ImmutableMap;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Date;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "SendEventForReturnToOrder",
    description = "Send an {" + PROP_EVENT_NAME + "} event to the Order ",
    produces = {
        @EventInfo(
            eventName = PROP_EVENT_NAME,
            entityType = EventInfoVariables.EVENT_TYPE,
            entitySubtype = EventInfoVariables.EVENT_SUBTYPE,
            status = EventInfoVariables.EVENT_STATUS)
    },
    accepts = {
        @EventInfo(
            entityType = EntityType.RETURN_ORDER
        )
    }
)
@ParamString(name = PROP_EVENT_NAME,
    description = "Event name for outgoing event")
@Slf4j
public class SendEventForReturnToOrder implements Rule {

    private static String CLASS_NAME = SendEventForReturnToOrder.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        final Event incomingEvent = context.getEvent();
        final String accountId = incomingEvent.getAccountId();
        final String entityRef = incomingEvent.getEntityRef();
        final String retailerId = incomingEvent.getRetailerId();
        log.info(MessageFormat.format("{0} - rule: {1} incoming event {2}", logPrefix, CLASS_NAME, incomingEvent));
        try {
            validateContext(context);

            GetReturnOrderQuery.Data returnOrderData =
                (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                    .ref(entityRef)
                    .retailer(RetailerId.builder().id(retailerId).build())
                    .includeAttributes(false)
                    .includeReturnOrderItems(false)
                    .includePickupLocation(false)
                    .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                    .build());

            if (returnOrderData == null || returnOrderData.returnOrder() == null) {
                throw new IllegalArgumentException(String.format("ReturnOrder was not found with ref: %s", entityRef));
            }

            GetReturnOrderQuery.ReturnOrder returnOrder = returnOrderData.returnOrder();
            final String orderRef = returnOrder.order().ref();
            GetOrderByRefQuery.Data orderData =
                (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
                    .ref(orderRef)
                    .includeCustomer(false)
                    .includeAttributes(false)
                    .includeFulfilmentChoice(false)
                    .includeFulfilments(false)
                    .includeOrderItems(false)
                    .includeArticles(false)
                    .includeConsignmentArticles(false)
                    .includeFinancialTransactions(false)
                    .build());
            if (orderData == null || orderData.order() == null) {
                throw new IllegalArgumentException(format("Order was not found by ref %s", orderRef));
            }
            Event event = createEvent(context, orderData.order());
            log.info(
                MessageFormat.format("{0} - rule: {1}, sending event {2} with name {3}", logPrefix, CLASS_NAME,
                    event, context.getProp(PROP_EVENT_NAME)));
            context.action().sendEvent(event);
        } catch (Exception ex) {
            log.error(MessageFormat.format("{0} - rule: {1} exception message: {2}, stackTrace: {3}",
                logPrefix, CLASS_NAME,
                ex.getMessage() == null ? ex.toString() : ex.getMessage(), Arrays.toString(ex.getStackTrace())));
            throw new RubixException(400,
                format("Rule: " + CLASS_NAME + " - exception message: %s. Stacktrace: %s", ex.getMessage(),
                    Arrays.toString(ex.getStackTrace())));
        }
    }

    private Event createEvent(Context context,
        GetOrderByRefQuery.Order order) {
        return Event.builder()
            .name(context.getProp(PROP_EVENT_NAME))
            .scheduledOn(new Date())
            .entitySubtype(order.type())
            .accountId(context.getEvent().getAccountId())
            .retailerId(context.getEvent().getRetailerId())
            .entityType(EntityType.ORDER)
            .rootEntityRef(order.ref())
            .rootEntityType(EntityType.ORDER)
            .entityRef(order.ref())
            .entityId(order.id())
            .attributes(ImmutableMap.<String, Object>builder()
                .putAll(context.getEvent().getAttributes()).build())
            .build();
    }

    private void validateContext(Context context) {
        if (isEmpty(context.getProp(PROP_EVENT_NAME))) {
            throw new IllegalArgumentException("No " + PROP_EVENT_NAME + " is provided");
        }
    }

}
