package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.PROP_STATUS;

import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderWithFulfilmentsQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderWithFulfilmentsQuery.Edge1;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.EventUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.List;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "SendEventIfAnyReturnFulfilmentInStatuses",
    description = "Sends event {" + PROP_EVENT_NAME + "} if any Return Fulfilment has statuses {" + PROP_STATUS + "}.",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(name = PROP_EVENT_NAME,
    description = "Event name to be send")
@ParamString(name = PROP_STATUS,
    description = "List of statuses")
@Slf4j
public class SendEventIfAnyReturnFulfilmentInStatuses implements Rule {

    private static final String CLASS_NAME = SendEventIfAnyReturnFulfilmentInStatuses.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME);
        final String eventName = context.getProp(PROP_EVENT_NAME);
        final List<String> statuses = context.getPropList(PROP_STATUS, String.class);
        if (statuses == null || statuses.isEmpty()) {
            String message = MessageFormat.format("{0} - Statuses are empty!", logPrefix);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        final String returnOrderRef = context.getEntity().getRef();
        final String retailerId = context.getEvent().getRetailerId();

        RuleUtils.isNotNull(returnOrderRef, retailerId);

        GetReturnOrderWithFulfilmentsQuery.Data returnOrder =
            (GetReturnOrderWithFulfilmentsQuery.Data) context.api().query(GetReturnOrderWithFulfilmentsQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .includeAttributes(false)
                .includeReturnOrderItems(false)
                .includePickupLocation(false)
                .includeReturnOrderFulfilments(true)
                .returnOrderFulfilmentCount(DEFAULT_PAGE_SIZE)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null
            || returnOrder.returnOrder().returnOrderFulfilments() == null
            || returnOrder.returnOrder().returnOrderFulfilments().edges() == null
            || returnOrder.returnOrder().returnOrderFulfilments().edges().isEmpty()) {
            String message =
                MessageFormat.format("{0} - Return Order with ref {1} has not been found or it has no fulfilments!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        boolean hasStatus = false;
        for (Edge1 edge : returnOrder.returnOrder().returnOrderFulfilments().edges()) {
            String status = edge.node().status();
            if (statuses.contains(status)) {
                hasStatus = true;
                log.info(MessageFormat.format("{0} - Return Order has Fulfilment in status {1}",
                    logPrefix, status));
                break;
            }
        }

        if (hasStatus) {
            log.info(MessageFormat.format("{0} - Sending event {1} to the same entity", logPrefix, eventName));
            EventUtils.forwardEvent(context, eventName);
        }
    }
}
