package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_DELAY;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_EXCHANGE_ORDER_REF;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_EXCHANGE_ORDER_EXISTED;

import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Attribute;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamEnum;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.ParamTimePeriod;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.AttributeUtils;
import com.fluentretail.se.plugins.util.Constants.BooleanOptions;
import com.fluentretail.se.plugins.util.EventUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@RuleInfo(
    name = "SendEventIfExchangeOrderExistedOrNot",
    description = "Sends event {" + PROP_EVENT_NAME + "} if Exchange Order existed is {" + PROP_EXCHANGE_ORDER_EXISTED
        + "} with delay {" + PROP_DELAY + "}.",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER),
        @EventInfo(entityType = RETURN_FULFILMENT)
    }
)
@ParamString(
    name = PROP_EVENT_NAME,
    description = "Event name to be sent")
@ParamEnum(
    name = PROP_EXCHANGE_ORDER_EXISTED,
    description = "Exchange Order existence",
    options = BooleanOptions.class
)
@ParamTimePeriod(
    name = PROP_DELAY,
    description = "Number of seconds to be delayed"
)
@Slf4j
public class SendEventIfExchangeOrderExistedOrNot implements Rule {

    private static final String CLASS_NAME = SendEventIfExchangeOrderExistedOrNot.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        final String retailerId = context.getEvent().getRetailerId();

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME, PROP_EXCHANGE_ORDER_EXISTED);
        final String eventName = context.getProp(PROP_EVENT_NAME);
        final BooleanOptions exchangeExisted = context.getProp(PROP_EXCHANGE_ORDER_EXISTED, BooleanOptions.class);
        final int delay = context.getProp(PROP_DELAY, Integer.class);

        final boolean isOrderEntity = context.getEvent().getEntityType().equals(RETURN_ORDER);
        final String returnOrderRef = isOrderEntity ? context.getEvent().getEntityRef() :
            context.getEvent().getRootEntityRef();
        GetReturnOrderQuery.Data returnOrder =
            (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .includeAttributes(true)
                .includeReturnOrderItems(false)
                .includePickupLocation(false)
                .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null) {
            String message = MessageFormat.format("{0} - Return Order with ref {1} has not been found!",
                logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        Optional<Attribute> optional = AttributeUtils.getAttribute(ATTR_EXCHANGE_ORDER_REF,
            returnOrder.returnOrder().attributes());
        final BooleanOptions exchangeOrderExisted = (optional.isPresent() &&
            StringUtils.isNotEmpty(optional.get().value().toString())) ? BooleanOptions.TRUE : BooleanOptions.FALSE;
        if (exchangeExisted == exchangeOrderExisted) {
            log.info(MessageFormat.format("{0} - Sending event {1} to the same entity", logPrefix, eventName));
            if (delay > 0) {
                EventUtils.forwardScheduledEvent(context, eventName, delay);
            } else {
                EventUtils.forwardEvent(context, eventName);
            }
        }
    }
}
