package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.PROP_ATTRIBUTE_NAME;
import static com.fluentretail.se.plugins.util.Constants.PROP_ATTRIBUTE_VALUE;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;

import com.fluentcommerce.se.common.graphql.queries.returnfulfilment.GetReturnFulfilmentQuery;
import com.fluentcommerce.se.common.graphql.queries.returnfulfilment.GetReturnFulfilmentQuery.Attribute;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.EventUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@RuleInfo(
    name = "SendEventIfReturnFulfilmentAttributeEqualsTo",
    description = "Sends event {" + PROP_EVENT_NAME + "} if attribute {" + PROP_ATTRIBUTE_NAME + "} is equal to {"
        + PROP_ATTRIBUTE_VALUE + "}",
    accepts = {
        @EventInfo(entityType = RETURN_FULFILMENT)
    }
)
@ParamString(name = PROP_EVENT_NAME,
    description = "Event name to be send")
@ParamString(name = PROP_ATTRIBUTE_NAME,
    description = "Attribute name to be checked")
@ParamString(name = PROP_ATTRIBUTE_VALUE,
    description = "Attribute value to be checked with")
@Slf4j
public class SendEventIfReturnFulfilmentAttributeEqualsTo implements Rule {

    private static final String CLASS_NAME = SendEventIfReturnFulfilmentAttributeEqualsTo.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        final String fulfilmentRef = context.getEntity().getRef();
        final String returnOrderRef = context.getEvent().getRootEntityRef();
        final String retailerId = context.getEvent().getRetailerId();

        if (!RuleUtils.isNotNull(fulfilmentRef, returnOrderRef, retailerId)) {
            String message = MessageFormat
                .format("{0} - Some values are null: Fulfilment Ref - {1}, Order Ref - {2}, RetailerId - {3}",
                    logPrefix, fulfilmentRef, returnOrderRef, retailerId);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME, PROP_ATTRIBUTE_NAME, PROP_ATTRIBUTE_VALUE);

        GetReturnFulfilmentQuery.Data fulfilment = (GetReturnFulfilmentQuery.Data) context.api()
            .query(GetReturnFulfilmentQuery.builder()
                .ref(fulfilmentRef)
                .returnOrderRef(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .includeAttributes(true)
                .build());

        if (fulfilment == null || fulfilment.returnFulfilment() == null) {
            String message = MessageFormat.format("{0} - Fulfilment with Ref {1} has not been found!",
                logPrefix, fulfilmentRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        final String attributeName = context.getProp(PROP_ATTRIBUTE_NAME);
        final String attributeValue = findAttributeValue(fulfilment.returnFulfilment().attributes(), attributeName);
        if (StringUtils.isEmpty(attributeValue)) {
            log.info(MessageFormat.format("{0} - Attribute with name {1} has not been found!",
                logPrefix, attributeName));
            return;
        }

        if (attributeValue.equals(context.getProp(PROP_ATTRIBUTE_VALUE))) {
            Event event = EventUtils.createEvent(context, context.getProp(PROP_EVENT_NAME), null);
            log.info(MessageFormat.format("{0} - Sending event: {1}", logPrefix, event));
            context.action().sendEvent(event);
        } else {
            log.info(MessageFormat.format("{0} - Attribute value {1} is not equal to {2}",
                logPrefix, attributeValue, context.getProp(PROP_ATTRIBUTE_VALUE)));
        }
    }

    private String findAttributeValue(List<Attribute> attrs, String attributeName) {
        if (attrs == null || attrs.isEmpty()) {
            return null;
        }
        for (Attribute attr : attrs) {
            if (attributeName.equals(attr.name())) {
                return attr.value().toString();
            }
        }
        return null;
    }
}
