package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_DELAY;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_TYPE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.RMA;

import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemEdge;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemNode;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.ParamTimePeriod;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.EventUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "SendEventIfReturnOrderIsNotRMA",
    description = "Sends event {" + PROP_EVENT_NAME + "} if Return Order doesn't have RMA items with delay {"
        + PROP_DELAY + "}.",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(
    name = PROP_EVENT_NAME,
    description = "Event name to be sent")
@ParamTimePeriod(
    name = PROP_DELAY,
    description = "Number of seconds to be delayed"
)
@Slf4j
public class SendEventIfReturnOrderIsNotRMA implements Rule {

    private static final String CLASS_NAME = SendEventIfReturnOrderIsNotRMA.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME);
        final String eventName = context.getProp(PROP_EVENT_NAME);
        final int delay = context.getProp(PROP_DELAY, Integer.class);

        final String returnOrderRef = context.getEntity().getRef();
        GetReturnOrderQuery.Data returnOrder =
            (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .includeAttributes(true)
                .includeReturnOrderItems(true)
                .includePickupLocation(false)
                .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null
            || returnOrder.returnOrder().returnOrderItems() == null
            || returnOrder.returnOrder().returnOrderItems().edges() == null
            || returnOrder.returnOrder().returnOrderItems().edges().isEmpty()) {
            String message =
                MessageFormat.format("{0} - Return Order with ref {1} has not been found or it has no items!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        final Set<String> orderItemRefs = returnOrder.returnOrder().returnOrderItems().edges().stream()
            .filter(it -> it.node() != null && it.node().orderItem() != null)
            .map(it -> it.node().orderItem().ref())
            .collect(Collectors.toSet());

        final String orderRef = returnOrder.returnOrder().order().ref();
        GetOrderByRefQuery.Data order = (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
            .ref(orderRef)
            .includeOrderItems(true)
            .includeAttributes(true)
            .includeCustomer(false)
            .includeFulfilments(false)
            .includeFulfilmentChoice(false)
            .includeArticles(false)
            .includeConsignmentArticles(false)
            .includeFinancialTransactions(false)
            .orderItemCount(DEFAULT_PAGE_SIZE)
            .build());

        if (order == null || order.order() == null || order.order().items() == null
            || order.order().items().itemEdges() == null
            || order.order().items().itemEdges().isEmpty()) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found or it has no items!",
                logPrefix, orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        boolean isRma = false;
        for (ItemEdge item : order.order().items().itemEdges()) {
            ItemNode node = item.itemNode();
            if (orderItemRefs.contains(node.ref())) {
                List<String> attrs = (node.attributes() == null) ? null : node.attributes().stream()
                    .filter(at -> ATTR_RETURN_TYPE.equals(at.name()))
                    .map(at -> at.value().toString())
                    .collect(Collectors.toList());
                if (attrs != null && attrs.size() == 1 && RMA.equals(attrs.get(0))) {
                    isRma = true;
                    break;
                }
            }
        }

        if (!isRma) {
            log.info(MessageFormat.format("{0} - Sending event {1} to the same entity", logPrefix, eventName));
            if (delay > 0) {
                EventUtils.forwardScheduledEvent(context, eventName, delay);
            } else {
                EventUtils.forwardEvent(context, eventName);
            }
        }
    }
}
