package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;

import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderWithFulfilmentsQuery;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.EventUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "SendEventToAllReturnFulfilments",
    description = "Sends event {" + PROP_EVENT_NAME + "} to all return fulfilments of current Return Order",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(name = PROP_EVENT_NAME,
    description = "Event name to be send")
@Slf4j
public class SendEventToAllReturnFulfilments implements Rule {

    private static final String CLASS_NAME = SendEventToAllReturnFulfilments.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME);

        final String returnOrderRef = context.getEntity().getRef();
        final String eventName = context.getProp(PROP_EVENT_NAME);

        GetReturnOrderWithFulfilmentsQuery.Data order =
            (GetReturnOrderWithFulfilmentsQuery.Data) context.api().query(GetReturnOrderWithFulfilmentsQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .includeAttributes(false)
                .includeReturnOrderItems(false)
                .includePickupLocation(false)
                .includeReturnOrderFulfilments(true)
                .returnOrderFulfilmentCount(DEFAULT_PAGE_SIZE)
                .build());

        if (order == null || order.returnOrder() == null || order.returnOrder().returnOrderFulfilments() == null) {
            String message =
                MessageFormat.format("{0} - Return Order with ref {1} has not been found or it has no fulfilments!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        order.returnOrder().returnOrderFulfilments().edges().forEach(edge -> {
            EventUtils.forwardEventToSubEntity(context, eventName, edge.node().ref(), RETURN_FULFILMENT, DEFAULT);
        });
    }
}
