package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.PROP_SETTING;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_EXCHANGE_ORDER_REF;

import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery.Attribute;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.AttributeUtils;
import com.fluentretail.se.plugins.util.Constants.EntityType;
import com.fluentretail.se.plugins.util.RuleUtils;
import com.fluentretail.se.plugins.util.SettingUtils;
import com.google.common.collect.ImmutableMap;
import java.text.MessageFormat;
import java.util.Date;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@RuleInfo(
    name = "SendEventToExchangeOrderIfReturnOrderInStatus",
    description = "Sends event with name {" + PROP_EVENT_NAME +
        "} to Exchange Order if Return Order is in status specified in setting {" + PROP_SETTING + "}",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(name = PROP_EVENT_NAME,
    description = "Event to be sent")
@ParamString(name = PROP_SETTING,
    description = "Setting name that contains status name")

@Slf4j
public class SendEventToExchangeOrderIfReturnOrderInStatus implements Rule {

    private static final String CLASS_NAME = SendEventToExchangeOrderIfReturnOrderInStatus.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME, PROP_SETTING);
        final String eventName = context.getProp(PROP_EVENT_NAME);
        final String settingName = context.getProp(PROP_SETTING);

        final String returnOrderRef = context.getEvent().getEntityRef();
        final String retailerId = context.getEvent().getRetailerId();
        GetReturnOrderQuery.Data returnOrder =
            (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .includeAttributes(true)
                .includeReturnOrderItems(false)
                .includePickupLocation(false)
                .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null) {
            String message = MessageFormat.format("{0} - Return Order with ref {1} has not been found!",
                logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        Optional<Attribute> optional = AttributeUtils.getAttribute(ATTR_EXCHANGE_ORDER_REF, returnOrder.returnOrder()
            .attributes());
        if (!optional.isPresent()) {
            log.info(MessageFormat.format("{0} - There is no Exchange Order for the Return Order with ref {1}",
                logPrefix, returnOrderRef));
            return;
        }
        final String exchangeOrderRef = (String) optional.get().value();

        final String settingValue = SettingUtils.getSettingValue(context, settingName, "");
        if (StringUtils.isEmpty(settingValue)) {
            String message = MessageFormat.format("{0} - Setting with name {1} has not been found!",
                logPrefix, settingName);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        if (settingValue.equals(returnOrder.returnOrder().status())) {
            GetOrderByRefQuery.Data orderData =
                (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
                    .ref(exchangeOrderRef)
                    .includeCustomer(false)
                    .includeAttributes(false)
                    .includeFulfilmentChoice(false)
                    .includeFulfilments(false)
                    .includeOrderItems(false)
                    .includeArticles(false)
                    .includeConsignmentArticles(false)
                    .includeFinancialTransactions(false)
                    .build());

            if (orderData == null || orderData.order() == null) {
                String message = MessageFormat.format("{0} - Exchange Order with ref {1} has not been found!",
                    logPrefix, exchangeOrderRef);
                log.error(message);
                throw new RuleExecutionException(message, context.getEvent());
            }
            log.info(MessageFormat.format("{0} - Sending event with name {1} to Order with ref {2}",
                logPrefix, eventName, exchangeOrderRef));
            Event event = createEvent(context, orderData.order());
            context.action().sendEvent(event);
        }
    }

    private Event createEvent(Context context,
        GetOrderByRefQuery.Order order) {
        return Event.builder()
            .name(context.getProp(PROP_EVENT_NAME))
            .scheduledOn(new Date())
            .entitySubtype(order.type())
            .accountId(context.getEvent().getAccountId())
            .retailerId(context.getEvent().getRetailerId())
            .entityType(EntityType.ORDER)
            .rootEntityRef(order.ref())
            .rootEntityType(EntityType.ORDER)
            .entityRef(order.ref())
            .entityId(order.id())
            .attributes(ImmutableMap.<String, Object>builder()
                .putAll(context.getEvent().getAttributes()).build())
            .build();
    }
}
