package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_DELAY;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;

import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.ParamTimePeriod;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "SendEventToReturnOrderFromFulfilment",
    description = "Sends event {" + PROP_EVENT_NAME + "} to Return Order with delay {" + PROP_DELAY + "}",
    accepts = {
        @EventInfo(entityType = RETURN_FULFILMENT)
    }
)
@ParamString(name = PROP_EVENT_NAME,
    description = "Event name to be send")
@ParamTimePeriod(name = PROP_DELAY,
    description = "Delay of the event")
@Slf4j
public class SendEventToReturnOrderFromFulfilment implements Rule {

    private static final String CLASS_NAME = SendEventToReturnOrderFromFulfilment.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        RuleUtils.validateContextRefRootRef(context);
        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_EVENT_NAME);

        final int delay = context.getProp(PROP_DELAY, Integer.class);
        final String eventName = context.getProp(PROP_EVENT_NAME);
        final String retailerId = context.getEvent().getRetailerId();
        final String returnOrderRef = context.getEvent().getRootEntityRef();

        Event event = Event.builder()
            .retailerId(retailerId)
            .name(eventName)
            .entityType(RETURN_ORDER)
            .entityRef(returnOrderRef)
            .entitySubtype(DEFAULT)
            .rootEntityType(RETURN_ORDER)
            .rootEntityRef(returnOrderRef)
            .scheduledOn(new Date(LocalDateTime.now().plus(delay, ChronoUnit.SECONDS).atZone(ZoneId.systemDefault())
                .toInstant().toEpochMilli()))
            .build();
        log.info(MessageFormat.format("{0} - Sending event: {1}", logPrefix, event));
        context.action().sendEvent(event);
    }
}
