package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.INVENTORY_CATALOGUE;
import static com.fluentretail.se.plugins.util.Constants.ITEMS;
import static com.fluentretail.se.plugins.util.Constants.PROP_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.PROP_INVENTORY_CATALOGUE_REF;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_VALID_RETURN_CONDITION_LIST;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_VALID_RETURN_FULFILMENT_STATUS_LIST;

import com.fluentcommerce.se.common.graphql.queries.returnfulfilment.GetReturnFulfilmentQuery;
import com.fluentcommerce.se.common.graphql.queries.returnfulfilment.GetReturnFulfilmentQuery.Data;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderWithFulfilmentsQuery;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.gi.model.type.Items;
import com.fluentretail.se.plugins.util.RuleUtils;
import com.google.common.collect.Lists;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "SendEventToUpdateInventoryQuantityReturnOrder",
    description = "Send an event {" + PROP_EVENT_NAME + "} to inventory catalogue with ref {"
        + PROP_INVENTORY_CATALOGUE_REF + "} to increment the Return Order's items quantity "
        + "that have return condition {" + PROP_VALID_RETURN_CONDITION_LIST + "} and fulfilments in statuses {"
        + PROP_VALID_RETURN_FULFILMENT_STATUS_LIST + "}",
    accepts = {
        @EventInfo(entityType = "RETURN_ORDER")
    }
)
@ParamString(name = PROP_INVENTORY_CATALOGUE_REF,
    description = "The ref of Inventory catalogue")
@ParamString(name = PROP_EVENT_NAME,
    description = "The name of event to be triggered")
@ParamString(name = PROP_VALID_RETURN_CONDITION_LIST,
    description = "The list of accepted return conditions")
@ParamString(name = PROP_VALID_RETURN_FULFILMENT_STATUS_LIST,
    description = "The list of accepted return fulfilment statuses")
@Slf4j
public class SendEventToUpdateInventoryQuantityReturnOrder implements Rule {

    private static final String CLASS_NAME = SendEventToUpdateInventoryQuantityReturnOrder.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        final String responseEvent = context.getProp(PROP_EVENT_NAME);
        final String inventoryCatalogueRef = context.getProp(PROP_INVENTORY_CATALOGUE_REF);
        final String returnOrderRef = context.getEvent().getEntityRef();
        final String returnOrderRetailerId = context.getEvent().getRetailerId();

        final Set<String> acceptedConditions = getAcceptedReturnConditions(context);

        log.info(MessageFormat.format("{0} - Getting Return Order", logPrefix));
        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder = getReturnOrderQuery(context, returnOrderRef, null,
            returnOrderRetailerId, false, false, true, true).returnOrder();

        log.info(MessageFormat.format("{0} - Return Order={1}", logPrefix, returnOrder));
        List<GetReturnOrderWithFulfilmentsQuery.Edge> returnOrderItemEdges = returnOrder.returnOrderItems().edges();
        String returnDestination = returnOrder.destinationLocation().ref();

        final Map<String, Integer> quantitiesByOrderItemRefs =
            getItemsQuantityAccordingToFilfilments(context, returnOrderRef,
                returnOrder.returnOrderFulfilments().edges());

        Map<String, Integer> itemMap = new HashMap<>();
        for (GetReturnOrderWithFulfilmentsQuery.Edge edge : returnOrderItemEdges) {
            GetReturnOrderWithFulfilmentsQuery.Node returnOrderItem = edge.node();
            log.info(MessageFormat.format("{0} - ReturnOrderItem={1}", returnOrderItem));

            if (returnOrderItem.returnCondition() == null || returnOrderItem.returnCondition().value() == null ||
                acceptedConditions == null || !acceptedConditions.contains(returnOrderItem.returnCondition().value())) {
                log.info(MessageFormat.format("{0} - Order Item with ref {1} has bad condition!",
                    logPrefix, returnOrderItem.ref()));
                continue;
            }

            int returnQuantity = returnOrderItem.unitQuantity().quantity();
            if (quantitiesByOrderItemRefs != null && !quantitiesByOrderItemRefs.containsKey(returnOrderItem.ref())) {
                log.info(MessageFormat.format("{0} - Order Item with ref {1} has no accepted fulfilments!",
                    logPrefix, returnOrderItem.ref()));
                continue;
            } else {
                returnQuantity = Math.min(returnQuantity, quantitiesByOrderItemRefs.get(returnOrderItem.ref()));
            }
            itemMap.put(returnOrderItem.product().ref(),
                itemMap.getOrDefault(returnOrderItem.product().ref(), 0) + returnQuantity);
        }

        List<Items> items = Lists.newArrayList();
        itemMap.entrySet().stream().forEach(entry -> {
            items.add(Items.builder()
                .fulfilmentId(returnOrderRef)
                .skuRef(entry.getKey())
                .reserveQty(entry.getValue())
                .locationRef(returnDestination)
                .build());
        });

        if (items.isEmpty()) {
            log.warn(MessageFormat.format("{0} - There are no items to be returned", logPrefix));
            return;
        }

        Map<String, Object> attributes = new HashMap<>();
        attributes.put(ITEMS, items);
        Event event = Event.builder()
            .accountId(context.getEvent().getAccountId())
            .retailerId(context.getEvent().getRetailerId())
            .attributes(attributes)
            .entityRef(inventoryCatalogueRef)
            .entityType(INVENTORY_CATALOGUE)
            .rootEntityRef(inventoryCatalogueRef)
            .rootEntityType(INVENTORY_CATALOGUE)
            .entitySubtype(DEFAULT)
            .name(responseEvent)
            .scheduledOn(new Date(
                LocalDateTime.now().plus(1, ChronoUnit.SECONDS).atZone(ZoneId.systemDefault()).toInstant()
                    .toEpochMilli()))
            .build();
        log.info(MessageFormat.format("{0} - sending Event:{1}", event));
        context.action().sendEvent(event);
    }

    private GetReturnOrderWithFulfilmentsQuery.Data getReturnOrderQuery(Context context, String returnOrderRef,
        String returnOrderItemCursor, String retailerId, boolean includeAttributes, boolean includePickupLocation,
        boolean includeReturnOrderItems, boolean includeReturnFulfilments) {
        GetReturnOrderWithFulfilmentsQuery returnOrderQuery = GetReturnOrderWithFulfilmentsQuery.builder()
            .ref(returnOrderRef)
            .includeAttributes(includeAttributes)
            .includePickupLocation(includePickupLocation)
            .includeReturnOrderItems(includeReturnOrderItems)
            .includeReturnOrderFulfilments(includeReturnFulfilments)
            .returnOrderItemCount(DEFAULT_PAGE_SIZE)
            .returnOrderFulfilmentCount(DEFAULT_PAGE_SIZE)
            .returnOrderItemCursor(returnOrderItemCursor)
            .retailer(RetailerId.builder()
                .id(retailerId)
                .build())
            .build();
        return (GetReturnOrderWithFulfilmentsQuery.Data) context.api().query(returnOrderQuery);
    }

    private Set<String> getAcceptedReturnConditions(Context context) {
        final List<String> conditions = context.getPropList(PROP_VALID_RETURN_CONDITION_LIST, String.class);
        if (conditions == null || conditions.isEmpty()) {
            return null;
        }
        return new TreeSet<>(conditions);
    }

    private Map<String, Integer> getItemsQuantityAccordingToFilfilments(Context context, String returnOrderRef,
        List<GetReturnOrderWithFulfilmentsQuery.Edge1> fulfilments) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        if (fulfilments == null || fulfilments.isEmpty()) {
            return null;
        }
        List<String> statuses = context.getPropList(PROP_VALID_RETURN_FULFILMENT_STATUS_LIST, String.class);
        if (statuses != null && statuses.isEmpty()) {
            return null;
        }
        Map<String, Integer> quantitiesByOrderItemRefs = new HashMap<>();

        for (GetReturnOrderWithFulfilmentsQuery.Edge1 edge : fulfilments) {
            GetReturnFulfilmentQuery.ReturnFulfilment fulfilment =
                ((Data) context.api().query(GetReturnFulfilmentQuery.builder()
                    .ref(edge.node().ref())
                    .returnOrderRef(returnOrderRef)
                    .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                    .includeAttributes(false)
                    .includePickupLocation(false)
                    .includeReturnOrderItems(true)
                    .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                    .build()))
                    .returnFulfilment();
            if (fulfilment == null || fulfilment.returnFulfilmentItems() == null) {
                log.warn(
                    MessageFormat.format("{0} - Fulfilment with ref {1} has not been found or it has no items!",
                        logPrefix, edge.node().ref()));
                continue;
            }
            if (!statuses.contains(fulfilment.status())) {
                continue;
            }
            for (GetReturnFulfilmentQuery.Edge item : fulfilment.returnFulfilmentItems().edges()) {
                String orderItemRef = item.node().returnOrderItem().ref();
                Integer quantity = item.node().unitQuantity().quantity();
                quantitiesByOrderItemRefs.put(orderItemRef,
                    quantitiesByOrderItemRefs.getOrDefault(orderItemRef, 0) + quantity);
            }
        }
        return quantitiesByOrderItemRefs.isEmpty() ? null : quantitiesByOrderItemRefs;
    }
}