package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.ATTR_SUBSTITUTION_PRICE;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.PROP_STATUS;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_REMAIN_REFUND_AMOUNT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_BALANCE_REFUND_AMOUNT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_ORIGINAL_REFUND_AMOUNT;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_PRICE;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_TAX_PRICE;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_TOTAL_PRICE;
import static com.fluentretail.se.plugins.util.Constants.SUBSTITUTION_TOTAL_TAX_PRICE;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fluentcommerce.se.common.graphql.mutations.returnorder.UpdateReturnOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.Attribute2;
import com.fluentcommerce.se.common.graphql.queries.returnfulfilment.GetReturnFulfilmentQuery;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderWithFulfilmentsQuery;
import com.fluentcommerce.se.common.graphql.type.AmountTypeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.UpdateReturnOrderInput;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.AttributeUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "UpdateTotalAmountForReturnOrder",
    description = "Updates total amount for Return Order taking into account only Fulfilments in statuses {" +
        PROP_STATUS + "}",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(name = PROP_STATUS,
    description = "List of statuses")
@Slf4j
public class UpdateTotalAmountForReturnOrder implements Rule {

    private static final String CLASS_NAME = UpdateTotalAmountForReturnOrder.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event: {1}", logPrefix, context.getEvent()));

        final List<String> statuses = context.getPropList(PROP_STATUS, String.class);
        if (statuses == null || statuses.isEmpty()) {
            String message = MessageFormat.format("{0} - Statuses are empty!", logPrefix);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder = getReturnOrder(context);

        Map<String, Integer> productQuantities = new HashMap<>();
        boolean isAllFulfilmentsAreValid = processFulfilments(context, new TreeSet<>(statuses), returnOrder,
            productQuantities);

        if (isAllFulfilmentsAreValid) {
            log.info(MessageFormat.format("{0} - All Fulfilments are in valid statuses. No updates are required",
                logPrefix));
            return;
        }
        double priceCorrection = getPriceCorrection(returnOrder);

        GetOrderByRefQuery.Order order = getOrder(context, returnOrder.order().ref());
        double subTotalAmount = 0.0;
        double totalTaxAmount = 0.0;

        for (GetOrderByRefQuery.ItemEdge edge : order.items().itemEdges()) {
            final GetOrderByRefQuery.ItemNode item = edge.itemNode();
            final String productRef = item.product().asVariantProduct().ref();
            if (productQuantities.containsKey(productRef)) {
                Integer quantity = productQuantities.get(productRef);
                Double itemPrice = getItemPrice(item, priceCorrection);
                Double itemTax = getItemTaxPrice(item, quantity, priceCorrection);
                if (itemPrice == null || itemPrice <= 0.0) {
                    String message = MessageFormat.format("{0} - Price is not set for product {1} in orderItem {2}",
                        logPrefix, productRef, item.ref());
                    log.error(message);
                    throw new RuleExecutionException(message, context.getEvent());
                }

                subTotalAmount += quantity * itemPrice;
                totalTaxAmount += itemTax;
            }
        }

        log.info(MessageFormat.format("{0} - Recalculated subTotalAmount: {1}, totalTaxAmount: {2}",
            logPrefix, subTotalAmount, totalTaxAmount));
        context.action().mutation(UpdateReturnOrderMutation.builder()
            .input(UpdateReturnOrderInput.builder()
                .ref(returnOrder.ref())
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .subTotalAmount(AmountTypeInput.builder().amount(subTotalAmount).build())
                .totalTax(AmountTypeInput.builder().amount(totalTaxAmount).build())
                .totalAmount(AmountTypeInput.builder().amount(subTotalAmount + totalTaxAmount).build())
                .build())
            .build());
    }

    private Double getItemPrice(GetOrderByRefQuery.ItemNode item, double correction) {
        Double itemPrice = item.price();
        if (itemPrice != null && itemPrice > 0) {
            return correctAmountValue(itemPrice, correction);
        }
        Double totalPrice = item.totalPrice();
        if (totalPrice != null && totalPrice > 0) {
            return correctAmountValue(totalPrice / item.quantity(), correction);
        }
        itemPrice = getPriceForSubstituteItem(item, SUBSTITUTION_PRICE);
        if (itemPrice != null && itemPrice > 0) {
            return correctAmountValue(itemPrice, correction);
        }
        totalPrice = getPriceForSubstituteItem(item, SUBSTITUTION_TOTAL_PRICE);
        if (totalPrice != null && totalPrice > 0) {
            return correctAmountValue(totalPrice / item.quantity(), correction);
        }
        return null;
    }

    private Double getItemTaxPrice(GetOrderByRefQuery.ItemNode item, Integer returnQuantity, double correction) {
        Double taxPrice = item.taxPrice();
        if (taxPrice != null && taxPrice > 0.0) {
            return correctAmountValue(taxPrice, correction);
        }
        taxPrice = item.totalTaxPrice();
        if (taxPrice != null && taxPrice > 0.0) {
            return correctAmountValue(taxPrice / item.quantity(), correction) * returnQuantity;
        }
        taxPrice = getPriceForSubstituteItem(item, SUBSTITUTION_TAX_PRICE);
        if (taxPrice != null && taxPrice > 0.0) {
            return correctAmountValue(taxPrice, correction);
        }
        taxPrice = getPriceForSubstituteItem(item, SUBSTITUTION_TOTAL_TAX_PRICE);
        if (taxPrice != null && taxPrice > 0.0) {
            return correctAmountValue(taxPrice / item.quantity(), correction) * returnQuantity;
        }

        return 0.0;
    }

    private Double getPriceForSubstituteItem(GetOrderByRefQuery.ItemNode item, String priceKey) {
        Optional<Attribute2> price = AttributeUtils.getAttribute(ATTR_SUBSTITUTION_PRICE, item.attributes());
        if (price.isPresent()) {
            ObjectMapper mapper = new ObjectMapper();
            Map<String, Double> value = mapper.convertValue(price.get().value(),
                new TypeReference<Map<String, Double>>() {
                });
            return value.getOrDefault(priceKey, null);
        }
        return null;
    }

    private GetReturnOrderWithFulfilmentsQuery.ReturnOrder getReturnOrder(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        final String returnOrderRef = context.getEntity().getRef();
        final String retailerId = context.getEvent().getRetailerId();

        RuleUtils.isNotNull(returnOrderRef, retailerId);

        GetReturnOrderWithFulfilmentsQuery.Data data =
            (GetReturnOrderWithFulfilmentsQuery.Data) context.api().query(GetReturnOrderWithFulfilmentsQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(retailerId).build())
                .includeAttributes(true)
                .includeReturnOrderItems(true)
                .includePickupLocation(false)
                .includeReturnOrderFulfilments(true)
                .returnOrderFulfilmentCount(DEFAULT_PAGE_SIZE)
                .build());

        if (data == null || data.returnOrder() == null) {
            String message = MessageFormat.format("{0} - Return Order with ref {1} has not been found!",
                logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder = data.returnOrder();

        if (returnOrder.returnOrderFulfilments() == null
            || returnOrder.returnOrderFulfilments().edges() == null
            || returnOrder.returnOrderFulfilments().edges().isEmpty()) {
            String message = MessageFormat.format("{0} - Return Order with ref {1} has no fulfilments!",
                logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        if (returnOrder.returnOrderItems() == null
            || returnOrder.returnOrderItems().edges() == null
            || returnOrder.returnOrderItems().edges().isEmpty()) {
            String message = MessageFormat.format("{0} - Return Order with ref {1} has no items!",
                logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        return returnOrder;
    }

    private boolean processFulfilments(Context context, Set<String> validStatuses,
        GetReturnOrderWithFulfilmentsQuery.ReturnOrder returnOrder, Map<String, Integer> productQuantities) {
        boolean allInValidStatuses = true;

        for (GetReturnOrderWithFulfilmentsQuery.Edge1 edge : returnOrder.returnOrderFulfilments().edges()) {
            GetReturnOrderWithFulfilmentsQuery.Node1 node = edge.node();
            if (!validStatuses.contains(node.status())) {
                allInValidStatuses = false;
                continue;
            }

            GetReturnFulfilmentQuery.Data data =
                (GetReturnFulfilmentQuery.Data) context.api().query(GetReturnFulfilmentQuery.builder()
                    .ref(node.ref())
                    .returnOrderRef(returnOrder.ref())
                    .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                    .includeAttributes(true)
                    .includeReturnOrderItems(true)
                    .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                    .build());

            if (data == null || data.returnFulfilment() == null
                || data.returnFulfilment().returnFulfilmentItems() == null
                || data.returnFulfilment().returnFulfilmentItems().edges() == null) {
                continue;
            }

            List<GetReturnFulfilmentQuery.Edge> items = data.returnFulfilment().returnFulfilmentItems().edges();
            for (GetReturnFulfilmentQuery.Edge item : items) {
                final String productRef = item.node().product().ref();
                final Integer quantity = item.node().unitQuantity().quantity();
                productQuantities.put(productRef, quantity + productQuantities.getOrDefault(productRef, 0));
            }
        }

        return allInValidStatuses;
    }

    private GetOrderByRefQuery.Order getOrder(Context context, String orderRef) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());

        GetOrderByRefQuery.Data data = (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
            .ref(orderRef)
            .includeOrderItems(true)
            .includeAttributes(true)
            .orderItemCount(DEFAULT_PAGE_SIZE)
            .build());

        if (data == null || data.order() == null || data.order().items() == null
            || data.order().items().itemEdges() == null || data.order().items().itemEdges().isEmpty()) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found or it has no items!",
                logPrefix, orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        return data.order();
    }

    private double getPriceCorrection(GetReturnOrderWithFulfilmentsQuery.ReturnOrder order) {
        for (GetReturnOrderWithFulfilmentsQuery.Attribute attribute : order.attributes()) {
            if (ATTR_REMAIN_REFUND_AMOUNT.equals(attribute.name())) {
                Map<String, Object> value = (Map<String, Object>) attribute.value();
                if (value.containsKey(PROP_ORIGINAL_REFUND_AMOUNT) && value.containsKey(PROP_BALANCE_REFUND_AMOUNT)) {
                    return (Double) value.get(PROP_BALANCE_REFUND_AMOUNT) / (Double) value.get(
                        PROP_ORIGINAL_REFUND_AMOUNT);
                }
            }
        }
        return 1.0;
    }

    private double correctAmountValue(double value, double correction) {
        return (correction < 1.0) ? BigDecimal.valueOf(value * correction)
            .setScale(2, RoundingMode.HALF_UP).doubleValue() : value;
    }
}
