package com.fluentcommerce.rule.order.returnorder;

import static com.fluentretail.se.plugins.util.Constants.ATTR_SUBSTITUTION_QUANTITY;
import static com.fluentretail.se.plugins.util.Constants.Attributes.Types.JSON;
import static com.fluentretail.se.plugins.util.Constants.DEFAULT_PAGE_SIZE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.RETURN_ORDER;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURNABLE_QTY;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_DATE_LIMIT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_REJECTION_REASONS;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ATTR_RETURN_TYPE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_CONFIRM_RETURN_ORDER_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.PROP_REJECT_RETURN_ORDER_EVENT_NAME;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.REJECTION_ITEM;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.REJECTION_PRODUCT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.REJECTION_REASON;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.RejectionReason.NOT_RETURNABLE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.RejectionReason.RETURN_EXPIRED;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.RejectionReason.WRONG_QUANTITY;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.DEFAULT;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.NONE;
import static com.fluentretail.se.plugins.util.Constants.ReturnOrder.ReturnType.RMA;

import com.fluentcommerce.se.common.graphql.mutations.returnorder.UpdateReturnOrderMutation;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemEdge;
import com.fluentcommerce.se.common.graphql.queries.order.GetOrderByRefQuery.ItemNode;
import com.fluentcommerce.se.common.graphql.queries.returnorder.GetReturnOrderQuery;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentcommerce.se.common.graphql.type.RetailerId;
import com.fluentcommerce.se.common.graphql.type.UpdateReturnOrderInput;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RuleExecutionException;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.se.plugins.util.EventUtils;
import com.fluentretail.se.plugins.util.RuleUtils;
import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;

@RuleInfo(
    name = "ValidateReturnOrderItems",
    description = "Validates Return Order items and send {" + PROP_CONFIRM_RETURN_ORDER_EVENT_NAME +
        "} event on success. Otherwise it sends {" + PROP_REJECT_RETURN_ORDER_EVENT_NAME + "} event.",
    accepts = {
        @EventInfo(entityType = RETURN_ORDER)
    }
)
@ParamString(name = PROP_CONFIRM_RETURN_ORDER_EVENT_NAME,
    description = "Confirm Return Order event")
@ParamString(name = PROP_REJECT_RETURN_ORDER_EVENT_NAME,
    description = "Reject Return Order event")
@Slf4j
public class ValidateReturnOrderItems implements Rule {

    private static final String CLASS_NAME = ValidateReturnOrderItems.class.getSimpleName();

    @Override
    public void run(Context context) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        log.info(MessageFormat.format("{0} - Incoming event:{1}", logPrefix, context.getEvent()));

        RuleUtils.validateRulePropsIsNotEmpty(context, PROP_CONFIRM_RETURN_ORDER_EVENT_NAME,
            PROP_REJECT_RETURN_ORDER_EVENT_NAME);

        final String confirmEvent = context.getProp(PROP_CONFIRM_RETURN_ORDER_EVENT_NAME);
        final String rejectEvent = context.getProp(PROP_REJECT_RETURN_ORDER_EVENT_NAME);

        final String returnOrderRef = context.getEntity().getRef();
        GetReturnOrderQuery.Data returnOrder =
            (GetReturnOrderQuery.Data) context.api().query(GetReturnOrderQuery.builder()
                .ref(returnOrderRef)
                .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                .includeAttributes(true)
                .includeReturnOrderItems(true)
                .includePickupLocation(false)
                .returnOrderItemCount(DEFAULT_PAGE_SIZE)
                .build());

        if (returnOrder == null || returnOrder.returnOrder() == null
            || returnOrder.returnOrder().returnOrderItems() == null) {
            String message =
                MessageFormat.format("{0} - Return Order with ref {1} has not been found or it has no items!",
                    logPrefix, returnOrderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        final String orderRef = returnOrder.returnOrder().order().ref();
        GetOrderByRefQuery.Data order = (GetOrderByRefQuery.Data) context.api().query(GetOrderByRefQuery.builder()
            .ref(orderRef)
            .includeOrderItems(true)
            .includeAttributes(true)
            .includeCustomer(false)
            .includeFulfilments(false)
            .includeFulfilmentChoice(false)
            .includeArticles(false)
            .includeConsignmentArticles(false)
            .includeFinancialTransactions(false)
            .orderItemCount(DEFAULT_PAGE_SIZE)
            .build());

        if (order == null || order.order() == null || order.order().items() == null) {
            String message = MessageFormat.format("{0} - Order with ref {1} has not been found or it has no items!",
                logPrefix, orderRef);
            log.error(message);
            throw new RuleExecutionException(message, context.getEvent());
        }

        List<Map<String, String>> rejections = new ArrayList<>();
        final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        boolean validationFailed = false;
        for (GetReturnOrderQuery.Edge edge : returnOrder.returnOrder().returnOrderItems().edges()) {
            ItemNode orderItem = findOrderItem(edge.node(), order.order().items());
            if (orderItem == null) {
                String message = MessageFormat.format("{0} - Order Item for product {1} has not been found!",
                    logPrefix, edge.node().product().ref());
                log.error(message);
                throw new RuleExecutionException(message, context.getEvent());
            }
            String returnType =
                getOrderItemAttributeValue(orderItem, ATTR_RETURN_TYPE, DEFAULT).toString().trim().toUpperCase();
            switch (returnType) {
                case DEFAULT:
                case RMA:
                    break;
                case "":
                    returnType = DEFAULT;
                    break;
                default:
                    returnType = NONE;
            }

            if (NONE.equals(returnType)) {
                // The Item is not returnable
                rejections.add(createRejectionReason(orderItem.ref(), orderItem.product().asVariantProduct().name(),
                    NOT_RETURNABLE));
                validationFailed = true;
                continue;
            }
            Object value = getOrderItemAttributeValue(orderItem, ATTR_RETURN_DATE_LIMIT, null);
            Date returnDateLimit = null;
            if (value != null) {
                try {
                    returnDateLimit = sdf.parse(value.toString());
                } catch (ParseException ignored) {
                    returnDateLimit = null;
                }
            }
            if (returnDateLimit != null && returnDateLimit.before(new Date())) {
                // Return Date has expired
                rejections.add(createRejectionReason(orderItem.ref(), orderItem.product().asVariantProduct().name(),
                    RETURN_EXPIRED));
                validationFailed = true;
                continue;
            }
            Object qty = getOrderItemAttributeValue(orderItem, ATTR_RETURNABLE_QTY, null);
            if (qty == null) {
                qty = orderItem.quantity() - Integer.parseInt(getOrderItemAttributeValue(orderItem,
                    ATTR_SUBSTITUTION_QUANTITY,
                    "0").toString());
            }
            String qtyValue = qty.toString();
            if (!qtyValue.matches("[0-9]+")) {
                String message = MessageFormat.format("{0} - ReturnableQty value {1} is bad!",
                    logPrefix, qty);
                log.error(message);
                throw new RuleExecutionException(message, context.getEvent());
            }
            if (Integer.parseInt(qtyValue) < edge.node().unitQuantity().quantity()) {
                // Return Quantity is wrong
                rejections.add(createRejectionReason(orderItem.ref(), orderItem.product().asVariantProduct().name(),
                    WRONG_QUANTITY));
                validationFailed = true;
            }
        }

        if (validationFailed) {
            context.action().mutation(UpdateReturnOrderMutation.builder()
                .input(UpdateReturnOrderInput.builder()
                    .ref(returnOrderRef)
                    .retailer(RetailerId.builder().id(context.getEvent().getRetailerId()).build())
                    .attributes(Collections.singletonList(AttributeInput.builder()
                        .name(ATTR_RETURN_REJECTION_REASONS)
                        .type(JSON)
                        .value(rejections)
                        .build()))
                    .build())
                .build());
        }

        Event event = validationFailed ?
            EventUtils.createEvent(context, rejectEvent,
                Collections.singletonMap(ATTR_RETURN_REJECTION_REASONS, rejections)) :
            EventUtils.createEvent(context, confirmEvent, null);
        log.info(MessageFormat.format("{0} - Sending event: {1}", context, event));
        context.action().sendEvent(event);
    }

    private ItemNode findOrderItem(GetReturnOrderQuery.Node returnItem, GetOrderByRefQuery.Items orderItems) {
        final boolean byItem = returnItem.orderItem() != null;
        final String key = byItem ? returnItem.orderItem().ref() : returnItem.product().ref();
        List<ItemEdge> items = orderItems.itemEdges().stream()
            .filter(it -> (byItem && key.equals(it.itemNode().ref())) || (!byItem && key.equals(
                it.itemNode().product().asVariantProduct().ref())))
            .collect(Collectors.toList());
        return items.size() == 1 ? items.get(0).itemNode() : null;
    }

    private Object getOrderItemAttributeValue(ItemNode item, String attributeName, Object defaultValue) {
        if (item.attributes() != null) {
            for (GetOrderByRefQuery.Attribute2 attr : item.attributes()) {
                if (attributeName.equals(attr.name())) {
                    return attr.value();
                }
            }
        }
        return defaultValue;
    }

    private Map<String, String> createRejectionReason(String orderItem, String product, String reason) {
        Map<String, String> rejectionReason = new HashMap<>();
        rejectionReason.put(REJECTION_ITEM, orderItem);
        rejectionReason.put(REJECTION_PRODUCT, product);
        rejectionReason.put(REJECTION_REASON, reason);
        return rejectionReason;
    }
}
