package com.fluentretail.se.plugins.util;

import static com.fluentretail.se.plugins.util.Constants.ErrorCodes.ERROR_CODE_400;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fluentcommerce.se.common.graphql.type.AttributeInput;
import com.fluentretail.api.model.attribute.Attribute;
import com.fluentretail.api.model.location.Address;
import com.fluentretail.rubix.exceptions.RubixException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;

@Slf4j
public class AttributeUtils {

    private static final String NAME_METHOD = "name";
    private static final String VALUE_METHOD = "value";

    private static final ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        mapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
        mapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
    }

    public static <T> String getAttributeValue(String attributeName, List<T> attributeList) {
        Optional<Method> nameMethod = getAttributeMethod(NAME_METHOD, attributeList);
        Optional<Method> valueMethod = getAttributeMethod(VALUE_METHOD, attributeList);
        if (CollectionUtils.isNotEmpty(attributeList) && nameMethod.isPresent() && valueMethod.isPresent()) {
            for (T a : attributeList) {
                String name;
                try {
                    name = ((String) nameMethod.get().invoke(a, null));
                    boolean matched = name.equalsIgnoreCase(attributeName);
                    if (matched) {
                        Object value = valueMethod.get().invoke(a, null);
                        return value + "";
                    }
                } catch (Exception e) {
                    log.error("ERR:Retrieving attribute value for attr {}.", attributeName);
                }
            }
        }
        return null;
    }

    public static <T> Optional<T> getAttribute(String attributeName, List<T> attributeList) {
        Optional<Method> nameMethod = getAttributeMethod(NAME_METHOD, attributeList);
        if (CollectionUtils.isNotEmpty(attributeList) && nameMethod.isPresent()) {
            return attributeList.stream()
                .filter(e -> {
                    try {
                        return ((String) nameMethod.get().invoke(e, null)).equalsIgnoreCase(attributeName);
                    } catch (Exception ex) {
                        throw new RubixException(ERROR_CODE_400, ex.getCause().toString());
                    }
                })
                .findFirst();
        } else {
            return Optional.empty();
        }
    }

    private static <T> Optional<Method> getAttributeMethod(String methodName, List<T> attributeList) {
        if (CollectionUtils.isNotEmpty(attributeList)) {
            return getAttributeMethodFromAttribute(methodName, attributeList.get(0));
        }
        return Optional.empty();
    }

    /**
     * To get  method with the given methodName
     *
     * @param <T>
     * @param methodName
     * @param attribute
     * @return
     */
    private static <T> Optional<Method> getAttributeMethodFromAttribute(String methodName, T attribute) {
        try {
            return Optional.ofNullable(attribute.getClass().getMethod(methodName, null));
        } catch (NoSuchMethodException e) {
            throw new RubixException(ERROR_CODE_400, e.getCause().toString());
        }
    }

    public static List<AttributeInput> of(Map<String, Object> attributes) {
        List<AttributeInput> attributeList = null;
        if (attributes != null) {
            attributeList = new ArrayList<>();
            for (Map.Entry<String, Object> attribute : attributes.entrySet()) {
                Object value = attribute.getValue();
                if (value == null) {
                    continue;
                }

                String type = "OBJECT";
                if (value instanceof String) {
                    if (((String) value).length() > Attribute.MAX_STRING_LENGTH) {
                        type = "TEXT";
                    } else {
                        type = "STRING";
                    }
                } else if (value instanceof Integer) {
                    type = "INTEGER";
                } else if (value instanceof Boolean) {
                    type = "BOOLEAN";
                } else if (value instanceof Address) {
                    type = "ADDRESS";
                }
                attributeList.add(
                    AttributeInput.builder().name(attribute.getKey()).value(attribute.getValue()).type(type).build());
            }
        }

        return attributeList;
    }

}
