package com.fluentretail.se.plugins.util;

import static com.fluentretail.se.plugins.util.Constants.EntityType.ARTICLE;
import static com.fluentretail.se.plugins.util.Constants.EntityType.CONSIGNMENT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.FULFILMENT;
import static com.fluentretail.se.plugins.util.Constants.EntityType.FULFILMENT_OPTIONS;
import static com.fluentretail.se.plugins.util.Constants.EntityType.FULFILMENT_PLAN;
import static com.fluentretail.se.plugins.util.Constants.EntityType.LOCATION;
import static com.fluentretail.se.plugins.util.Constants.EntityType.ORDER;
import static com.fluentretail.se.plugins.util.Constants.EntityType.WAVE;

import com.fluentretail.api.model.EntityType;
import com.fluentretail.api.v2.model.Entity;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@Slf4j
public class EventUtils {

    private static final String CLASS_NAME = EventUtils.class.getSimpleName();

    // ensure inline event is not scheduled, fetch entity to ensure latest entity status, and entitysubtype is set
    // if incoming state different to current state, schedule (WORKAROUND Rubix Bug)
    public static Event createEvent(Context context, String eventName, Map<String, Object> attributes) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());

        Entity entity = context.getEntity();
        String entityStatus = entity == null ? context.getEvent().getEntityStatus() : entity.getStatus();
        String entitySubtype = entity == null ? context.getEvent().getEntitySubtype() : entity.getType();

        entityStatus = checkAndRetrieveV1EntityStatus(context, entityStatus);

        Event.Builder eventBuilder = context.getEvent().toBuilder()
            .scheduledOn(
                !StringUtils.equalsIgnoreCase(entityStatus, context.getEvent().getEntityStatus()) ? new Date() : null)
            .entityStatus(entityStatus)
            .entitySubtype(entitySubtype)
            .name(eventName);
        if (attributes != null) {
            eventBuilder.attributes(attributes);
        }
        Event event = eventBuilder.build();
        log.info(
            "{} - Sending new event: {}. context.getEntity null?:{}."
                + " entityStatus:{} != context.getEvent().getEntityStatus():{}",
            logPrefix, event, entity, entityStatus, context.getEvent().getEntityStatus());
        return event;
    }

    public static void forwardEvent(Context context, String eventName) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());

        Event newEvent = context.getEvent().toBuilder()
            .name(eventName)
            .scheduledOn(null)
            .build();

        log.info(logPrefix + " - Forwarding event: {}", newEvent);

        log.info(MessageFormat.format("Forwarding event: {0}", newEvent));

        context.action().sendEvent(newEvent);
    }

    public static void forwardEventToSubEntity(Context context, String eventName, String ref, String type,
        String subType) {
        final String logPrefix = RuleUtils.buildLogPrefix(CLASS_NAME, context.getEvent());
        Event newEvent = context.getEvent().toBuilder()
            .name(eventName)
            .entityRef(ref)
            .entityType(type)
            .entitySubtype(subType)
            .scheduledOn(new Date())
            .build();
        log.info(logPrefix + " - Forwarding event to sub entity: {}", newEvent);
        log.info(MessageFormat.format("Forwarding event to sub entity: {0}", newEvent));
        context.action().sendEvent(newEvent);
    }

    public static void forwardScheduledEvent(Context context, String eventName, int delayedSeconds) {
        if (delayedSeconds > 0) {
            Event currentEvent = context.getEvent();

            Event scheduledEvent = Event.builder()
                .name(eventName)
                .accountId(currentEvent.getAccountId())
                .retailerId(currentEvent.getRetailerId())
                .entityId(currentEvent.getEntityId())
                .entityRef(currentEvent.getEntityRef())
                .entityType(currentEvent.getEntityType())
                .entitySubtype(currentEvent.getEntitySubtype())
                .rootEntityRef(currentEvent.getRootEntityRef())
                .rootEntityType(currentEvent.getRootEntityType())
                .rootEntityId(currentEvent.getRootEntityId())
                .attributes(currentEvent.getAttributes())
                .scheduledOn(new Date(
                    LocalDateTime.now().plus(delayedSeconds, ChronoUnit.SECONDS).atZone(ZoneId.systemDefault())
                        .toInstant()
                        .toEpochMilli()))
                .build();

            context.action().sendEvent(scheduledEvent);
        }
    }

    private static String checkAndRetrieveV1EntityStatus(Context context, String entityStatus) {

        com.fluentretail.api.model.Entity v1Entity;
        switch (context.getEvent().getEntityType()) {
            case ORDER:
            case FULFILMENT:
            case CONSIGNMENT:
            case ARTICLE:
            case FULFILMENT_OPTIONS:
            case FULFILMENT_PLAN:
            case LOCATION:
            case WAVE:
                v1Entity = context.api().getLegacyClient()
                    .getEntity(EntityType.of(context.getEvent().getEntityType()), context.getEvent().getEntityId());
                log.info(
                    "[{}] [COMMON-V2] [{}]- Got Legacy Entity {}, Legacy Entity Status: {}, Event Entity Status: {}",
                    context.getEvent().getAccountId(), EventUtils.class.getSimpleName(), v1Entity, v1Entity.getStatus(),
                    entityStatus);
                entityStatus =
                    entityStatus.equalsIgnoreCase(v1Entity.getStatus()) ? entityStatus : v1Entity.getStatus();
                break;
            default:
                break;
        }
        return entityStatus;
    }
}
