package com.fluentretail.se.plugins.util.helpers;

import static com.fluentretail.se.plugins.util.Constants.DEFAULT_TAX_TYPE;
import static com.fluentretail.se.plugins.util.helpers.SettingHelpers.searchSettingInAllContext;
import static com.fluentretail.se.plugins.util.helpers.SettingHelpers.transformSettingResponseToMap;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fluentcommerce.se.common.graphql.queries.setting.GetSettingsByNameQuery;
import com.fluentcommerce.se.common.graphql.type.TaxTypeInput;
import com.fluentretail.rubix.v2.context.Context;
import java.text.MessageFormat;
import java.util.Map;
import javax.annotation.Nullable;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class TaxHelper {

    public static final String TAX_TYPE_COUNTRY = "country";
    public static final String TAX_TYPE_GROUP = "group";
    public static final String TAX_TYPE_TARIFF = "tariff";


    public static double getTariffPercentageForTariff(Context context, @Nullable String tariff) {
        String defaultTariff =
            isEmpty(tariff) ? generateTaxTypeInputFromSetting(context, null).tariff().trim().toUpperCase() : tariff;
        log.info("The defaultTariff is {}", defaultTariff);
        log.info(MessageFormat.format("The defaultTariff is {0}", defaultTariff));

        GetSettingsByNameQuery.Data response = searchSettingInAllContext(context, defaultTariff);
        log.info("The percentageForTariff found was {}", response);
        log.info(MessageFormat.format("The percentageForTariff found was {0}", response));

        if (response == null || response.settings() == null || response.settings().edges().isEmpty() || isEmpty(
            response.settings().edges().get(0).node().value())) {
            throw new IllegalArgumentException(format("Missing setting for tariff = [%s]", defaultTariff));
        }

        try {
            return (Double.parseDouble(response.settings().edges().get(0).node().value()) * 100);
        } catch (Exception ex) {
            throw new IllegalArgumentException(format("Error Double.parsing(tariff). For tariff = %s, "
                    + "string value of percentage = %s. Exception = %s", defaultTariff,
                response.settings().edges().get(0).node().value(), ex.getMessage()));
        }
    }

    /**
     * This method will query the taxType from the setting and wrap it in the TaxTypeInput object.
     *
     * @param context     rule context to be able to trigger gql queries
     * @param settingName OPTIONAL, if not provided will take default value of "DEFAULT_TAX_TYPE"
     * @return TaxTypeInput object, for the 'settingName' introduced or DEFAULT_TAX_TYPE in case was not provided.
     */
    public static TaxTypeInput generateTaxTypeInputFromSetting(Context context, @Nullable String settingName) {
        String country;
        String group;
        String tariff;
        Map<String, Object> taxSettingMap = transformSettingResponseToMap(
            searchSettingInAllContext(context, isEmpty(settingName) ? DEFAULT_TAX_TYPE : settingName));

        if (taxSettingMap.containsKey(DEFAULT_TAX_TYPE)) {
            try {
                ObjectNode lob = (ObjectNode) taxSettingMap.get(DEFAULT_TAX_TYPE);
                country = lob.get(TAX_TYPE_COUNTRY).asText();
                group = lob.get(TAX_TYPE_GROUP).asText();
                tariff = lob.get(TAX_TYPE_TARIFF).asText();

            } catch (Exception e) {
                throw new IllegalArgumentException(
                    format("setting '%s' incorrectly configured, expect LOB/JSON value with 3 keys: ["
                        + "country, group, tariff]. Exception: %s", DEFAULT_TAX_TYPE, e.getMessage()));
            }
            if (isEmpty(country) || isEmpty(group) || isEmpty(tariff)) {
                throw new IllegalArgumentException("Empty values: [country, group, tariff] all should have values.");
            }

            return TaxTypeInput.builder().country(country).group(group).tariff(tariff).build();
        } else {
            throw new IllegalArgumentException(
                format("The setting [%s] was not found. Value found: %s", DEFAULT_TAX_TYPE, taxSettingMap));
        }
    }


    public static Double calculatePaidPriceFromPrice(Double price, double tariffPercentage) {
        if (price < 0 || tariffPercentage < 0) {
            throw new IllegalArgumentException(
                format("The amount should be positive. [price] = %s, [tariffPercentage] = %s", price,
                    tariffPercentage));
        }

        return price * ((100.0 + tariffPercentage) / 100);
    }

}