import { FC } from 'react';
import { Grid, makeStyles, Typography } from '@material-ui/core';
import { TemplateRegistry } from 'mystique/registry/TemplateRegistry';
import { useI18n } from 'mystique/hooks/useI18n';
import { useDefaultTariffValue } from '../../hooks/useDefaultTariffValue';

interface PriceDifferenceProps {
  difference: number;
  currency: string;
  zeroPayment?: boolean;
}

const PriceDifference: FC<PriceDifferenceProps> = ({
  difference,
  currency,
  zeroPayment = false,
}) => {
  const { translate } = useI18n();
  const tariffValue = useDefaultTariffValue();
  const classes = useStyles();

  return (
    <>
      <Grid item className={classes.diff}>
        <Grid container direction="row">
          <Grid item>
            <Typography>
              {translate('fc.se.priceDifference.priceAdjustment')}
            </Typography>
          </Grid>
          <Grid item>
            <Typography
              className={
                difference == 0
                  ? classes.black
                  : difference < 0
                    ? classes.green
                    : classes.red
              }
            >
              {(difference > 0 ? '+' : '') +
                TemplateRegistry.render(
                  `{{currency ${difference} '${currency}'}}`,
                )}
            </Typography>
          </Grid>
        </Grid>
        <Grid container direction="row">
          <Grid item>
            <Typography>
              {translate('fc.se.priceDifference.taxAdjustment')}
            </Typography>
          </Grid>
          <Grid item>
            <Typography
              className={
                difference == 0
                  ? classes.black
                  : difference < 0
                    ? classes.green
                    : classes.red
              }
            >
              {(difference > 0 ? '+' : '') +
                TemplateRegistry.render(
                  `{{currency ${(difference * tariffValue).toFixed(
                    2,
                  )} '${currency}'}}`,
                )}
            </Typography>
          </Grid>
        </Grid>
        <Grid container direction="row">
          <Grid item>
            <Typography>
              {translate('fc.se.priceDifference.totalPayment')}
            </Typography>
          </Grid>
          <Grid item>
            <Typography
              className={
                difference == 0 || zeroPayment
                  ? classes.black
                  : difference < 0
                    ? classes.green
                    : classes.red
              }
            >
              {TemplateRegistry.render(
                `{{currency ${
                  zeroPayment
                    ? 0
                    : Math.abs(
                      (difference as number) + difference * tariffValue,
                    ).toFixed(2)
                } '${currency}'}}`,
              ) +
                (difference != 0 && !zeroPayment
                  ? ' ' +
                    translate(
                      difference < 0
                        ? 'fc.se.priceDifference.refund'
                        : 'fc.se.priceDifference.payment',
                    )
                  : '')}
            </Typography>
          </Grid>
        </Grid>
      </Grid>
    </>
  );
};

export default PriceDifference;

const useStyles = makeStyles(() => ({
  diff: {
    alignContent: 'center',
  },
  green: {
    marginLeft: '10px',
    color: 'green',
  },
  red: {
    marginLeft: '10px',
    color: 'red',
  },
  black: {
    marginLeft: '10px',
    color: 'black',
  },
}));

