package com.fluentcommerce.connect.custom.transform;

import com.fluentcommerce.connect.core.fulfilmentoption.FulfilmentOptionTransformationChain;
import com.fluentcommerce.connect.core.fulfilmentoption.FulfilmentOptionTransformationStep;
import com.fluentcommerce.connect.core.fulfilmentoption.data.*;
import com.fluentcommerce.connect.core.handler.context.FluentContext;
import com.fluentcommerce.connect.custom.mapper.AttributeMapper;
import com.fluentcommerce.connect.custom.service.LocationService;
import com.fluentcommerce.graphql.queries.location.GetLocationQuery;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotNull;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

@Component
@Slf4j
@FulfilmentOptionTransformationStep(id = "pickup-location-enrichment", priority = 1)
public class PickupLocationTransformation implements FulfilmentOptionTransformationChain<FulfilmentRequest, FulfilmentResponse> {

    final private LocationService locationService;

    private static final String PICKUP_LOCATION_REF = "pickupLocationRef";
    private static final String PICKUP_LOCATION_ADDRESS = "pickupLocationAddress";
    private static final String PICKUP_LOCATION_OPENING_TIME = "pickupLocationOpeningTime";
    private static final DateTimeFormatter INPUT_DATE_TIME_FORMAT = DateTimeFormatter.ofPattern("Hmm");
    private static final DateTimeFormatter OUTPUT_DATE_TIME_FORMAT = DateTimeFormatter.ofPattern("HH:mm");

    public PickupLocationTransformation(final LocationService locationService) {
        this.locationService = locationService;
    }

    @Override
    public FulfilmentResponse transform(final FulfilmentRequest request, final FulfilmentResponse response, final FluentContext context) {

        final FulfilmentResponseDataCreateFulfilmentOption createFulfilmentOption = response.getData().getCreateFulfilmentOption();

        final List<FulfilmentPlan> edges = CollectionUtils.emptyIfNull(response.getData().getCreateFulfilmentOption().getPlans().getEdges())
                .stream().map(edge -> {

                    final FulfilmentPlanNode node = edge.getNode();
                    final List<FluentAttribute> attributes = node.getAttributes() == null ? new ArrayList<>() : edge.getNode().getAttributes();

                    final Optional<FluentAttribute> optPickupLocationAttribute = attributes.stream()
                            .filter(attribute -> PICKUP_LOCATION_REF.equalsIgnoreCase(attribute.getName())).findFirst();
                    optPickupLocationAttribute.ifPresent(fluentAttribute -> enrichResponse(context, attributes, fluentAttribute));

                    return FulfilmentPlan.builder()
                            .node(FulfilmentPlanNode.builder()
                                    .id(node.getId())
                                    .ref(node.getRef())
                                    .status(node.getStatus())
                                    .type(node.getType())
                                    .eta(node.getEta())
                                    .splitCount(node.getSplitCount())
                                    .fulfilments(node.getFulfilments())
                                    .attributes(attributes).build())
                            .build();
                }).toList();

        return FulfilmentResponse.builder()
                .data(FulfilmentResponseData.builder()
                        .createFulfilmentOption(FulfilmentResponseDataCreateFulfilmentOption.builder()
                                .id(createFulfilmentOption.getId())
                                .ref(createFulfilmentOption.getRef())
                                .type(createFulfilmentOption.getType())
                                .retailerId(createFulfilmentOption.getRetailerId())
                                .orderType(createFulfilmentOption.getOrderType())
                                .locationRef(createFulfilmentOption.getLocationRef())
                                .status(createFulfilmentOption.getStatus())
                                .createdOn(createFulfilmentOption.getCreatedOn())
                                .updatedOn(createFulfilmentOption.getUpdatedOn())
                                .address(createFulfilmentOption.getAddress())
                                .products(createFulfilmentOption.getProducts())
                                .attributes(createFulfilmentOption.getAttributes())
                                .workflowRef(createFulfilmentOption.getWorkflowRef())
                                .workflowVersion(createFulfilmentOption.getWorkflowVersion())
                                .plans(FulfilmentResponseDataCreateFulfilmentOptionPlans.builder()
                                        .edges(edges)
                                        .build())
                                .build())
                        .build())
                .build();
    }

    private void enrichResponse(final FluentContext context, final List<FluentAttribute> attributes, final FluentAttribute optPickupLocationAttribute) {
        try {
            final Optional<GetLocationQuery.Location> optLocation = locationService.getLocation(context, (String) optPickupLocationAttribute.getValue());
            optLocation.ifPresent(location -> {
                if (location.primaryAddress() != null) {
                    attributes.add(FluentAttribute.builder()
                            .name(PICKUP_LOCATION_ADDRESS)
                            .type("JSON")
                            .value(AttributeMapper.fromGQL(Objects.requireNonNull(location.primaryAddress(), "primaryAddress is missing")))
                            .build());
                }

                if (location.openingSchedule() != null) {
                    attributes.add(FluentAttribute.builder()
                            .name(PICKUP_LOCATION_OPENING_TIME)
                            .type("STRING")
                            .value(transformOpeningTime(Objects.requireNonNull(location.openingSchedule(), "openingSchedule is missing")))
                            .build());
                }
            });
        } catch (final Exception e) {
            log.error("Fluent enrichment failed", e);
        }
    }

    private String transformOpeningTime(@NotNull GetLocationQuery.OpeningSchedule openingSchedule) {
        return "Mon: " + convertTimeRange(openingSchedule.monStart(), openingSchedule.monEnd())
                + ", Tue: " + convertTimeRange(openingSchedule.tueStart(), openingSchedule.tueEnd())
                + ", Wed: " + convertTimeRange(openingSchedule.wedStart(), openingSchedule.wedEnd())
                + ", Thurs: " + convertTimeRange(openingSchedule.thuStart(), openingSchedule.thuEnd())
                + ", Fri: " + convertTimeRange(openingSchedule.friStart(), openingSchedule.friEnd())
                + ", Sat: " + convertTimeRange(openingSchedule.satStart(), openingSchedule.satEnd())
                + ", Sun: " + convertTimeRange(openingSchedule.sunStart(), openingSchedule.sunEnd());
    }

    private String convertTimeRange(int start, int end) {
        if (start == 0 && end == 0) {
            return "Closed";
        } else {
            return convertTime(start) + " to " + convertTime(end);
        }
    }

    private String convertTime(int input) {
        if (input == 0) {
            return "00:00";
        }

        LocalTime time = LocalTime.parse(String.valueOf(input), INPUT_DATE_TIME_FORMAT);
        return time.format(OUTPUT_DATE_TIME_FORMAT);
    }
}
