package com.fluentcommerce.connect.controller;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fluentcommerce.connect.ConnectSDKBaseIntegrationTest;
import com.fluentcommerce.connect.core.fulfilmentoption.data.Address;
import com.fluentcommerce.connect.core.fulfilmentoption.data.FulfilmentRequest;
import com.fluentcommerce.connect.core.fulfilmentoption.data.FulfilmentResponse;
import com.fluentcommerce.connect.core.fulfilmentoption.data.ProductInfo;
import com.fluentcommerce.connect.core.utils.ConversionUtils;
import com.fluentcommerce.connect.core.web.advice.ApiError;
import com.google.common.collect.Maps;
import io.qameta.allure.Story;
import io.restassured.config.ObjectMapperConfig;
import io.restassured.config.RestAssuredConfig;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.testcontainers.shaded.com.google.common.collect.ImmutableList;

import java.util.Map;

import static com.fluentcommerce.connect.common.utils.IntegrationTestUtils.FLUENT_GRAPHQL_URL;
import static com.fluentcommerce.connect.common.utils.IntegrationTestUtils.assertURLCalledAndBodyMatchesJson;
import static com.fluentcommerce.connect.common.utils.IntegrationTestUtils.getResourceAsString;
import static com.fluentcommerce.connect.common.utils.IntegrationTestUtils.setupGraphQL;
import static io.restassured.RestAssured.given;
import static io.restassured.http.ContentType.JSON;
import static org.apache.http.HttpStatus.SC_BAD_REQUEST;
import static org.apache.http.HttpStatus.SC_OK;
import static org.apache.http.HttpStatus.SC_UNAUTHORIZED;

@Story("Fulfilment Option Controller")
class FulfilmentOptionControllerIntegrationTest extends ConnectSDKBaseIntegrationTest {

    private static final String VALID_API_TOKEN = "1283712983721983";
    private static final String ACCOUNT = "TEST";
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String FLUENT_ACCOUNT_HEADER = "fluent.account";

    private final RestAssuredConfig restAssuredConfig = RestAssuredConfig.config()
            .objectMapperConfig(ObjectMapperConfig.objectMapperConfig()
                    .jackson2ObjectMapperFactory((type, s) -> ConversionUtils.getObjectMapper()));

    @Test
    @DisplayName("Should return an enriched fulfilment option")
    void shouldReturnFulfilmentOption() throws JsonProcessingException {
        // given: a fulfilment option request
        setupGraphQL(fluentWireMockServer, ".*CreateSetting.*", "it/fc/create_setting_response.json");
        setupGraphQL(fluentWireMockServer, ".*updateSetting.*", "it/fc/update_setting_response.json");
        final FulfilmentRequest request = FulfilmentRequest.builder()
                .ref("abc123")
                .locationRef("1")
                .retailerId("1")
                .type("HD")
                .orderType("HD")
                .address(Address.builder()
                        .longitude(151.209_354)
                        .latitude(-33.884_844)
                        .build())
                .products(ImmutableList.of(ProductInfo.builder()
                                .requestedQuantity(4500)
                                .productRef("24-MB05")
                                .catalogueRef("PC:MASTER:1")
                        .build()))
                .attributes(ImmutableList.of())
                .build();
        setupGraphQL(fluentWireMockServer, ".*CreateFulfilmentOption.*", "it/fc/fulfilment-option/create_fulfilment_option_pre_enrichment_response.json");
        setupGraphQL(fluentWireMockServer, ".*GetLocation.*", "it/fc/location/get_location_response.json");

        final var expectedResponse = ConversionUtils.getObjectMapper()
                .readValue(getResourceAsString("it/fc/fulfilment-option/create_fulfilment_option_post_enrichment_response.json"), FulfilmentResponse.class);

        // when: making that request
        final Map<String, String> header = Maps.newHashMap();
        header.put(AUTHORIZATION_HEADER, "ApiKey " + VALID_API_TOKEN);
        header.put(FLUENT_ACCOUNT_HEADER, ACCOUNT);

        final var response = given()
            .config(restAssuredConfig)
            .port(getCurrentRandomPort())
            .accept(JSON)
            .contentType(JSON)
            .headers(header)
            .body(request)
            .when()
            .post("/api/v1/fluent-connect/fulfilment-option")
            .then()
            .statusCode(SC_OK)
            .extract()
            .response()
            .body()
            .as(FulfilmentResponse.class);

        // then: a CreateFulfilmentOption graphQL query should be sent
        assertURLCalledAndBodyMatchesJson(fluentWireMockServer, FLUENT_GRAPHQL_URL, ".*CreateFulfilmentOption.*", getResourceAsString("it/fc/fulfilment-option/create_fulfilment_option_request_body.json"));

        // and: a FulfilmentResponse should be returned
        Assertions.assertEquals(expectedResponse, response);
    }

    @Test
    @DisplayName("Should return 401 for invalid tokens")
    void shouldReturnInvalidApiToken() {
        // given: a FulfilmentRequest
        final FulfilmentRequest request = FulfilmentRequest.builder()
            .ref("abc123")
            .locationRef("1")
            .retailerId("1")
            .type("HD")
            .orderType("HD")
            .address(Address.builder()
                         .longitude(151.209_354)
                         .latitude(-33.884_844)
                         .build())
            .products(ImmutableList.of(ProductInfo.builder()
                                           .requestedQuantity(4500)
                                           .productRef("24-MB05")
                                           .catalogueRef("PC:MASTER:1")
                                           .build()))
            .attributes(ImmutableList.of())
            .build();

        // when: making a request with an invalid API token
        // then: a 401 should be returned
        final Map<String, String> header = Maps.newHashMap();
        header.put(AUTHORIZATION_HEADER, "ApiKey invalid");
        header.put(FLUENT_ACCOUNT_HEADER, ACCOUNT);

        given()
            .config(restAssuredConfig)
            .port(getCurrentRandomPort())
            .accept(JSON)
            .contentType(JSON)
            .headers(header)
            .body(request)
            .when()
            .post("/api/v1/fluent-connect/fulfilment-option")
            .then()
            .statusCode(SC_UNAUTHORIZED);
    }

    @Test
    @DisplayName("Should return 400 when fulfilment request isn't valid")
    void shouldReturnInvalidRequest() {
        // given: a fulfilment option request
        setupGraphQL(fluentWireMockServer, ".*CreateSetting.*", "it/fc/create_setting_response.json");
        setupGraphQL(fluentWireMockServer, ".*updateSetting.*", "it/fc/update_setting_response.json");
        final FulfilmentRequest request = FulfilmentRequest.builder()
                .ref("abc123")
                .locationRef("1")
                .retailerId("1")
                .type("HD")
                .orderType("HD")
                .address(Address.builder()
                        .longitude(151.209_354)
                        .latitude(-33.884_844)
                        .build())
                .products(ImmutableList.of(ProductInfo.builder()
                        .requestedQuantity(4500)
                        .productRef("24-MB05")
                        .catalogueRef("PC:MASTER:1")
                        .build()))
                .attributes(ImmutableList.of())
                .build();

        // and: an empty fulfilment option response
        setupGraphQL(fluentWireMockServer, ".*CreateFulfilmentOption.*", "it/fc/fulfilment-option/create_fulfilment_option_empty_response.json");

        // when: making a request
        // then: it should return a 400
        final Map<String, String> header = Maps.newHashMap();
        header.put(AUTHORIZATION_HEADER, "ApiKey " + VALID_API_TOKEN);
        header.put(FLUENT_ACCOUNT_HEADER, ACCOUNT);

        final ApiError errorMessage = given()
            .config(restAssuredConfig)
            .port(getCurrentRandomPort())
            .accept(JSON)
            .contentType(JSON)
            .headers(header)
            .body(request)
            .when()
            .post("/api/v1/fluent-connect/fulfilment-option")
            .then()
            .statusCode(SC_BAD_REQUEST)
            .extract()
            .response()
            .body()
            .as(ApiError.class);

        Assertions.assertEquals("400", errorMessage.getErrorCode());
        Assertions.assertEquals("Failed to create a valid FulfilmentResponse.", errorMessage.getErrorDescription());
    }
}
